package com.ark.dict;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import com.ark.net.urlconn.StringCallback;
import com.yanzhenjie.kalle.Kalle;
import com.yanzhenjie.kalle.simple.SimpleResponse;
import com.yanzhenjie.kalle.simple.cache.CacheMode;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

public final class ConfigMapLoader {

    private static final String KEY_CONFIG = "key_val_config";
    private static Map<String, String> mResponseMap = new HashMap<>();
    private static boolean success;
    private Map<String, String> mParamMap = new HashMap<>();

    private ConfigMapLoader() {
        Context context = Utils.getContext();
        mParamMap.put("appid", InnerUtils.encode32MD5(context.getPackageName()));
        mParamMap.put("package_name", context.getPackageName());
        mParamMap.put("sys", "android");
        mParamMap.put("os", "android");
        mParamMap.put("channel", InnerUtils.getChannel(context));
        mParamMap.put("appver", InnerUtils.getVersionName(context));
        mParamMap.put("lan", InnerUtils.getLanguage(context));
    }

    public static ConfigMapLoader getInstance() {
        return Holder.configMapLoader;
    }

    /**
     * 获取键值对信息
     *
     * @return Map
     */
    @NonNull
    public Map<String, String> getResponseMap() {
        if (!success) {
            request(null);
        }
        if (mResponseMap.isEmpty()) {
            String json = loadFromCache();
            try {
                mResponseMap = loadJsonToMap(json);
            } catch (Exception e) {
                //
            }
        }
        return mResponseMap;
    }

    @NonNull
    private Map<String, String> loadJsonToMap(String json) throws Exception {
        Map<String, String> mResponseMap = new HashMap<>();
        if (!TextUtils.isEmpty(json)) {
            JSONObject responseObject = new JSONObject(json);
            JSONObject resObject = responseObject.optJSONObject("res");
            JSONObject params = resObject.optJSONObject("params");
            JSONObject kvObj = params == null ? resObject : params;
            Iterator<String> iterator = kvObj.keys();
            String key;
            String value;
            while (iterator.hasNext()) {
                key = iterator.next();
                value = kvObj.optString(key);
                mResponseMap.put(key, value);
            }
        }
        return mResponseMap;
    }

    /**
     * 从缓存键值对中取出配置
     *
     * @return rawStr
     */
    @NonNull
    private String loadFromCache() {
        return ArkKv.getString(KEY_CONFIG, "");
    }

    /**
     * 请求配置
     */
    public void request(@Nullable final OnConfigListener onConfigListener) {
        Kalle.get(getRequestUrl())
                .tag(System.nanoTime())
                .cacheKey(KEY_CONFIG)
                .cacheMode(CacheMode.NETWORK_NO_THEN_READ_CACHE)
                .perform(new StringCallback<String>(Utils.getContext()) {
                    @Override
                    public void onResponse(SimpleResponse<String, String> response) {
                        String json = null;
                        if (response != null && response.isSucceed()) {
                            success = true;
                            json = response.succeed();
                            ArkKv.saveString(KEY_CONFIG, json);
                        }
                        if (TextUtils.isEmpty(json)) {
                            json = loadFromCache();
                        }
                        try {
                            mResponseMap = loadJsonToMap(json);
                            if (onConfigListener != null) {
                                onConfigListener.onSuccess();
                            }
                        } catch (Exception e) {
                            //
                            if (onConfigListener != null) {
                                onConfigListener.onFailure(e);
                            }
                        }
                    }
                });
    }

    /**
     * 是否是老后台,兼容不同的请求以及结果
     *
     * @return 是否是老后台
     */
    public boolean isOldServer() {
        String pkName = Utils.getContext().getPackageName();
        return TextUtils.equals(pkName, "com.emojifair.emoji")
                || TextUtils.equals(pkName, "com.emojifair.emoji")
                || TextUtils.equals(pkName, "com.novv.resshare")
                || TextUtils.equals(pkName, "com.androidesk")
                || TextUtils.equals(pkName, "com.lovebizhi.wallpaper")
                || TextUtils.equals(pkName, "com.androidesk.livewallpaper");
    }

    /**
     * 获取请求的url
     *
     * @return url
     */
    @NonNull
    public String getRequestUrl() {
        final StringBuilder builderUrl = new StringBuilder();
        if (isOldServer()) {
            builderUrl.append("http://service.kv.dandanjiang.tv/online/params");
        } else {
            builderUrl.append("http://service.kv.dandanjiang.tv/remote");
        }
        if (mParamMap.size() != 0) {
            builderUrl.append("?");
            for (Object object : mParamMap.entrySet()) {
                Map.Entry entry = (Map.Entry) object;
                String key = (String) entry.getKey();
                String val = (String) entry.getValue();
                builderUrl.append(key);
                builderUrl.append("=");
                builderUrl.append(val);
                builderUrl.append("&");
            }
            builderUrl.deleteCharAt(builderUrl.length() - 1);
        }
        return builderUrl.toString();
    }

    public interface OnConfigListener {

        void onSuccess();

        void onFailure(Exception e);
    }

    private static class Holder {

        private static ConfigMapLoader configMapLoader = new ConfigMapLoader();
    }
}


