package com.proximities.sdk;

import android.content.Context;
import android.os.StrictMode;
import android.util.Base64;

import com.proximities.sdk.bridge.OnRegisterIdentifierListener;
import com.proximities.sdk.bridge.OnRetrievePartnersListener;
import com.proximities.sdk.interfaces.ClosestPoiInterface;
import com.proximities.sdk.interfaces.FavoriteActionsInterface;
import com.proximities.sdk.interfaces.GetFavoritesInterface;
import com.proximities.sdk.interfaces.GetPoisInterface;
import com.proximities.sdk.interfaces.LogDatabaseInterface;
import com.proximities.sdk.interfaces.LogInterface;
import com.proximities.sdk.interfaces.LoyaltyInterface;
import com.proximities.sdk.interfaces.NfcInterface;
import com.proximities.sdk.interfaces.OnGetCampaignsByUserActionListener;
import com.proximities.sdk.interfaces.PartnerInterface;
import com.proximities.sdk.interfaces.QrCodeInterface;
import com.proximities.sdk.interfaces.TransmitterInterface;
import com.proximities.sdk.json.model.animation.AnimationData;
import com.proximities.sdk.json.model.closest_poi.BaseClosestPoi;
import com.proximities.sdk.json.model.feed.BaseFeed;
import com.proximities.sdk.json.model.log.BaseLogResponse;
import com.proximities.sdk.json.model.log.CampaignLogs;
import com.proximities.sdk.json.model.loyalty.BaseLoyalty;
import com.proximities.sdk.json.model.partner.BasePartner;
import com.proximities.sdk.json.model.partner.BasePoi;
import com.proximities.sdk.json.model.transmitter.BaseNfc;
import com.proximities.sdk.json.model.transmitter.BaseQrCode;
import com.proximities.sdk.json.model.transmitter.BaseTransmitterResponse;
import com.proximities.sdk.json.model.transmitter.Campaign;
import com.proximities.sdk.json.model.transmitter.Transmitter;
import com.proximities.sdk.json.model.transmitter_logs.TransmittersLogs;
import com.proximities.sdk.json.model.user.BaseRegisterUser;
import com.proximities.sdk.util.ProximitiesPrefs;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.lang.ref.WeakReference;
import java.net.URISyntaxException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.TimeUnit;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;

import okhttp3.Interceptor;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.logging.HttpLoggingInterceptor;
import retrofit2.Call;
import retrofit2.Callback;
import retrofit2.Response;
import retrofit2.Retrofit;
import retrofit2.converter.gson.GsonConverterFactory;

class ProximitiesNetworkManager {

    private static final String DEVICE = "android";
    private static final String API_VERSION = "3.0";
    private static final String APP_FIRST_LAUNCH = "1";

    private static final ProximitiesNetworkManager ourInstance = new ProximitiesNetworkManager();

    public static ProximitiesNetworkManager getInstance() {
        return ourInstance;
    }

    private WeakReference<Context> mContext;
    private ProximitiesApiService mClient;
    private boolean isInitialized = false;

    boolean isInitialized() {
        return isInitialized;
    }

    void init(Context context){
        isInitialized = true;
        mContext = new WeakReference<>(context);
        mClient = initRetrofitClient().create(ProximitiesApiService.class);
        StrictMode.ThreadPolicy policy = new StrictMode.ThreadPolicy.Builder().permitAll().build();
        StrictMode.setThreadPolicy(policy);
    }

    private Retrofit initRetrofitClient(){
        OkHttpClient.Builder okHttpClient = new OkHttpClient.Builder().addNetworkInterceptor(new Interceptor() {
            @Override
            public okhttp3.Response intercept(Chain chain) throws IOException {
                Request.Builder builder = chain.request().newBuilder();
                String signature = "";
                try {
                    signature = sign(mContext.get(), chain.request().method() + "&" + chain.request().url().toString());
                } catch (IOException | NoSuchAlgorithmException | InvalidKeyException | URISyntaxException e) {
                    e.printStackTrace();
                }
                Request.Builder request = builder.addHeader("signature", signature)
                        .addHeader("device", DEVICE)
                        .addHeader("language", Utils.getCurrentLanguage(mContext.get()))
                        .addHeader("appid", Utils.getAppId(mContext.get()))
                        .addHeader("apiversion", API_VERSION);

                if(ProximitiesPrefs.readAppFirstLaunch(mContext.get()).isEmpty()){
                    ProximitiesPrefs.writeAppFirstLaunch(mContext.get(), APP_FIRST_LAUNCH);
                    request.addHeader("appfirstlaunch", ProximitiesPrefs.readAppFirstLaunch(mContext.get()));
                }

                if (!ProximitiesPrefs.readUserEmail(mContext.get()).isEmpty()) {
                    request.addHeader("useridentifier", ProximitiesPrefs.readUserEmail(mContext.get()));
                } else if (!ProximitiesPrefs.readAnonymousId(mContext.get()).isEmpty()) {
                    request.addHeader("anonymousidentifier", ProximitiesPrefs.readAnonymousId(mContext.get()));
                } else {
                    ProximitiesPrefs.writeAnonymousId(mContext.get(), UUID.randomUUID().toString());
                    request.addHeader("anonymousidentifier", ProximitiesPrefs.readAnonymousId(mContext.get()));
                }
                return chain.proceed(request.build());
            }
        }).connectTimeout(5, TimeUnit.SECONDS)
                .readTimeout(5, TimeUnit.SECONDS)
                .writeTimeout(1, TimeUnit.MINUTES);

        if(BuildConfig.DEBUG){
            HttpLoggingInterceptor logging = new HttpLoggingInterceptor();
            logging.setLevel(HttpLoggingInterceptor.Level.HEADERS);
            okHttpClient.addInterceptor(logging);
        }

        Retrofit.Builder builder = new Retrofit.Builder()
                .baseUrl(BuildConfig.HOST)
                .addConverterFactory(GsonConverterFactory.create())
                .client(okHttpClient.build());

        return builder.build();
    }

    private String sign(Context ctx, String msgToSign) throws IOException, InvalidKeyException, NoSuchAlgorithmException, URISyntaxException {
        byte[] key = Utils.getAppSecret(ctx).getBytes("UTF-8");
        return signRequest(msgToSign, key);
    }

    private String signRequest(String resource, byte[] key) throws NoSuchAlgorithmException, InvalidKeyException,
            UnsupportedEncodingException, URISyntaxException {

        SecretKeySpec sha1Key = new SecretKeySpec(key, "HmacSHA1");

        Mac mac = Mac.getInstance("HmacSHA1");
        mac.init(sha1Key);

        byte[] sigBytes = mac.doFinal(resource.getBytes());

        return Base64.encodeToString(sigBytes, Base64.NO_WRAP);
    }

    ////// WS /////////

    void getPartnersAsync(final PartnerInterface callback, double latitude, double longitude, int radius) {
        Call<BasePartner> request = mClient.getPartnersInRange(latitude, longitude, radius);

        request.enqueue(new Callback<BasePartner>() {
            @Override
            public void onResponse(Call<BasePartner> call, Response<BasePartner> response) {
                if (response.isSuccessful()) {
                    callback.onGetPartnersSuccess(response.body());
                } else {
                    callback.onGetPartnersError();
                }
            }

            @Override
            public void onFailure(Call<BasePartner> call, Throwable t) {
                callback.onGetPartnersError();
            }
        });
    }

    BasePartner getPartnersSync(double latitude, double longitude, int radius) {
        Call<BasePartner> request = mClient.getPartnersInRange(latitude, longitude, radius);
        try {
            return request.execute().body();
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        }
    }

    void getPois(GetPoisInterface callback){
        Call<BasePoi> request = mClient.getPois();

        request.enqueue(new Callback<BasePoi>() {
            @Override
            public void onResponse(Call<BasePoi> call, Response<BasePoi> response) {
                if (response.isSuccessful()) {
                    callback.onGetPois(response.body());
                } else {
                    callback.onErrorGetPois("");
                }
            }

            @Override
            public void onFailure(Call<BasePoi> call, Throwable t) {
                callback.onErrorGetPois(t.getMessage());
            }
        });
    }

    void getClosestPoi(final ClosestPoiInterface callback, double latitude, double longitude) {
        Call<BaseClosestPoi> request = mClient.getClosestPoi(latitude, longitude);

        request.enqueue(new Callback<BaseClosestPoi>() {
            @Override
            public void onResponse(Call<BaseClosestPoi> call, Response<BaseClosestPoi> response) {
                if (response.isSuccessful()) {
                    callback.onGetClosestPoi(response.body().getData().getDistance());
                } else {
                    callback.onErrorGetClosestPoi();
                }
            }

            @Override
            public void onFailure(Call<BaseClosestPoi> call, Throwable t) {
                callback.onErrorGetClosestPoi();
            }
        });
    }
    void getFeedPartners(final OnRetrievePartnersListener callback, double latitude, double longitude, Map<String, Object> parameters) {
        Call<BaseFeed> request = mClient.getFeedPartners(latitude, longitude, parameters);
        request.enqueue(new Callback<BaseFeed>() {
            @Override
            public void onResponse(Call<BaseFeed> call, Response<BaseFeed> response) {
                if (response.isSuccessful()) {
                    callback.onRetrievePartners(response.body().getData().getPartners());
                } else {
                    callback.onRetrievePartnersError("");
                }
            }

            @Override
            public void onFailure(Call<BaseFeed> call, Throwable t) {
                callback.onRetrievePartnersError(t.getMessage());
            }
        });
    }

    void getCampaignsByUserAction(final OnGetCampaignsByUserActionListener callback, String userAction){
        Call<AnimationData> request = mClient.getCampaignsByUserAction(userAction);
        request.enqueue(new Callback<AnimationData>() {
            @Override
            public void onResponse(Call<AnimationData> call, Response<AnimationData> response) {
                if (response.isSuccessful()) {
                    callback.onGetCampaignsByUserAction((response.body().getData() != null) ? response.body().getData().getCampaigns() : new ArrayList<Campaign>());
                } else {
                    callback.onGetCampaignsByUserActionError();
                }
            }

            @Override
            public void onFailure(Call<AnimationData> call, Throwable t) {
                callback.onGetCampaignsByUserActionError();
            }
        });
    }

    void getFavorites(final GetFavoritesInterface callback){
        Call<AnimationData> request = mClient.getFavorites();
        request.enqueue(new Callback<AnimationData>() {
            @Override
            public void onResponse(Call<AnimationData> call, Response<AnimationData> response) {
                if (response.isSuccessful()) {
                    callback.onGetFavorites(response.body().getData().getCampaigns());
                } else {
                    callback.onGetFavoritesError();
                }
            }

            @Override
            public void onFailure(Call<AnimationData> call, Throwable t) {
                callback.onGetFavoritesError();
            }
        });
    }

    void postLogs(LogInterface callback, LogDatabaseInterface callbackForDb, CampaignLogs logs){
        Call<BaseLogResponse> request = mClient.postLogs(logs);
        request.enqueue(new Callback<BaseLogResponse>() {
            @Override
            public void onResponse(Call<BaseLogResponse> call, Response<BaseLogResponse> response) {
                if(response.isSuccessful()){
                    if(callback != null){
                        callback.onSendLogSuccess(response.body());
                    }
                    if(callbackForDb != null){
                        callbackForDb.onSendLogFromDatabaseSuccess();
                    }
                } else {
                    if(callback != null){
                        callback.onSendLogError(logs);
                    }
                    if(callbackForDb != null){
                        callbackForDb.onSendLogFromDatabaseError();
                    }
                }
            }

            @Override
            public void onFailure(Call<BaseLogResponse> call, Throwable t) {
                if(callback != null){
                    callback.onSendLogError(logs);
                }
                if(callbackForDb != null){
                    callbackForDb.onSendLogFromDatabaseError();
                }
            }
        });
    }

    void getTransmitter(TransmitterInterface callback, Transmitter transmitter, boolean isEddystone){
        Call<BaseTransmitterResponse> request;
        if(isEddystone){
            request = mClient.getEddystone(transmitter.getNamespace(), transmitter.getInstance());
        } else {
            request = mClient.getBeacon(transmitter.getUuid(), transmitter.getMajor(), transmitter.getMinor());
        }
        request.enqueue(new Callback<BaseTransmitterResponse>() {
            @Override
            public void onResponse(Call<BaseTransmitterResponse> call, Response<BaseTransmitterResponse> response) {
                BaseTransmitterResponse data = response.body();
                if(response.isSuccessful()){
                    if(data != null) {
                        callback.onGetBeaconContent(data.getData());
                    }
                } else {
                    callback.onGetBeaconError(transmitter);
                }
            }

            @Override
            public void onFailure(Call<BaseTransmitterResponse> call, Throwable t) {
                callback.onGetBeaconError(transmitter);
            }
        });
    }

    void getNfc(NfcInterface callback, String uuid, String major, String minor){
        Call<BaseNfc> request = mClient.getNfc(uuid, major, minor);
        request.enqueue(new Callback<BaseNfc>() {
            @Override
            public void onResponse(Call<BaseNfc> call, Response<BaseNfc> response) {
                if(response.isSuccessful() && response.body() != null){
                    List<Transmitter> transmitterList = response.body().getData().getTransmitters();
                    if(transmitterList != null && !transmitterList.isEmpty()){
                        callback.onGetNfcTransmitter(transmitterList.get(0));
                    } else {
                        callback.onNfcWithoutCampaign();
                    }
                } else {
                    callback.onGetNfcError();
                }
            }

            @Override
            public void onFailure(Call<BaseNfc> call, Throwable t) {
                callback.onGetNfcError();
            }
        });
    }

    void getQrCode(QrCodeInterface callback, String uuid, String major, String minor){
        Call<BaseQrCode> request = mClient.getQrCode(uuid, major, minor);
        request.enqueue(new Callback<BaseQrCode>() {
            @Override
            public void onResponse(Call<BaseQrCode> call, Response<BaseQrCode> response) {
                if(response.isSuccessful() && response.body() != null){
                    List<Transmitter> transmitterList = response.body().getData().getTransmitters();
                    if(transmitterList != null && !transmitterList.isEmpty()){
                        callback.onGetQrCodeTransmitter(transmitterList.get(0));
                    } else {
                        callback.onQrCodeWithoutCampaign();
                    }
                } else {
                    callback.onGetQrCodeError();
                }
            }

            @Override
            public void onFailure(Call<BaseQrCode> call, Throwable t) {
                callback.onGetQrCodeError();
            }
        });
    }

    void registerUser(OnRegisterIdentifierListener callback, Map<String, Object> body){
        Call<BaseRegisterUser> request = mClient.registerUser(body);
        request.enqueue(new Callback<BaseRegisterUser>() {
            @Override
            public void onResponse(Call<BaseRegisterUser> call, Response<BaseRegisterUser> response) {
                if(response.isSuccessful()){
                    callback.onRegisterIdentierSuccess();
                } else {
//                    ProximitiesPrefs.writeUserEmail(context, "");
                    callback.onErrorRegisteringIdentifier();
                }
            }

            @Override
            public void onFailure(Call<BaseRegisterUser> call, Throwable t) {
                //TODO Vérifier s'il est logique de remove le userEmail
//                ProximitiesPrefs.writeUserEmail(context, "");
                callback.onErrorRegisteringIdentifier();
            }
        });
    }

    void refreshLoyalty(LoyaltyInterface callback, int campaignId){
        Call<BaseLoyalty> request = mClient.refreshLoyaltyCampaign(campaignId);
        request.enqueue(new Callback<BaseLoyalty>() {
            @Override
            public void onResponse(Call<BaseLoyalty> call, Response<BaseLoyalty> response) {
                if(response.isSuccessful() && response.body().getData() != null){
                    callback.onRefreshLoyaltySuccess(response.body().getData().getCampaigns().get(0));
                } else {
                    callback.onRefreshLoyaltyError();
                }
            }

            @Override
            public void onFailure(Call<BaseLoyalty> call, Throwable t) {
                callback.onRefreshLoyaltyError();
            }
        });
    }

    void sendTransmitterLogs(TransmittersLogs logs){
        Call<Void> request = mClient.sendTransmitterLogs(logs);
        request.enqueue(new Callback<Void>() {
            @Override
            public void onResponse(Call<Void> call, Response<Void> response) {

            }

            @Override
            public void onFailure(Call<Void> call, Throwable t) {

            }
        });
    }

    void saveCampaign(FavoriteActionsInterface callback, int campaignId){
        Call<Void> request = mClient.saveCampaign(campaignId);
        request.enqueue(new Callback<Void>() {
            @Override
            public void onResponse(Call<Void> call, Response<Void> response) {
                if(response.isSuccessful()){
                    callback.onAddCampaignToFavorites();
                } else {
                    callback.onAddCampaignError();
                }
            }

            @Override
            public void onFailure(Call<Void> call, Throwable t) {
                callback.onAddCampaignError();
            }
        });
    }

    void deleteCampaign(FavoriteActionsInterface callback, int campaignId){
        Call<Void> request = mClient.deleteCampaign(campaignId);
        request.enqueue(new Callback<Void>() {
            @Override
            public void onResponse(Call<Void> call, Response<Void> response) {
                if(response.isSuccessful()){
                    callback.onDeleteCampaignFromFavorites();
                } else {
                    callback.onDeleteCampaignError();
                }
            }

            @Override
            public void onFailure(Call<Void> call, Throwable t) {
                callback.onDeleteCampaignError();
            }
        });
    }

}
