package com.proximities.sdk.request;

import com.proximities.sdk.json.WsHeader;
import com.android.volley.AuthFailureError;
import com.android.volley.NetworkResponse;
import com.android.volley.ParseError;
import com.android.volley.Request;
import com.android.volley.Response;
import com.android.volley.VolleyLog;
import com.android.volley.toolbox.HttpHeaderParser;
import com.google.gson.Gson;
import com.google.gson.JsonSyntaxException;

import java.io.UnsupportedEncodingException;
import java.util.Map;

import static com.proximities.sdk.util.LogUtils.LOGD;
import static com.proximities.sdk.util.LogUtils.makeLogTag;

/**
 * Created by Antoine Arnoult <arnoult.antoine@gmail.com> on 23/11/14.
 *
 * @see <a href="http://captechconsulting.com/blog/clinton-teegarden/volley-vs-retrofit-battle-royale-0">
 *          Battle royal
 *      </a>
 */
public class GsonRequest<T> extends Request<T> {

    private static final String TAG = makeLogTag(GsonRequest.class);

    private final Gson gson = new Gson();
    private String mRequestBody;
    private final Class<T> clazz;
    private final Map<String, String> headers;
    private final Response.Listener<T> listener;
    private static final String PROTOCOL_CHARSET = "utf-8";
    private boolean mCallFromGpsOrBeacon;  // add way to differentiate /partner and /partner/offer

    /**
     * Make a request and return a parsed object from JSON.
     *
     * @param method  type of request choosen in {@link com.android.volley.Request.Method}
     * @param url  URL of the request to make
     * @param clazz  Relevant class object, for Gson's reflection
     * @param headers  Map of request headers
     * @param listener  Listener if request is successful
     * @param errorListener  Listener if request fails
     */
    public GsonRequest(int method, String url, Class<T> clazz, Map<String, String> headers,
                       Response.Listener<T> listener, Response.ErrorListener errorListener) {
        super(method, url, errorListener);
        this.clazz = clazz;
        this.headers = headers;
        this.listener = listener;
        this.mRequestBody = null;
        // setRetryPolicy()
    }

    /**
     * Make a request and return a parsed object from JSON.
     *
     * @param method  type of request choosen in {@link com.android.volley.Request.Method}
     * @param url  URL of the request to make
     * @param clazz  Relevant class object, for Gson's reflection
     * @param headers  Map of request headers
     * @param listener  Listener if request is successful
     * @param errorListener  Listener if request fails
     * @param callFromGpsOrBeacon  Let we know if we call /partner or /partner/offer
     */
    public GsonRequest(int method, String url, Class<T> clazz, Map<String, String> headers,
                       Response.Listener<T> listener, Response.ErrorListener errorListener, boolean callFromGpsOrBeacon) {
        super(method, url, errorListener);
        this.clazz = clazz;
        this.headers = headers;
        this.listener = listener;
        this.mRequestBody = null;
        this.mCallFromGpsOrBeacon = callFromGpsOrBeacon;
    }

    /**
     * Make a request with body and return a parsed object from JSON.
     *
     * @param method  type of request choosen in {@link com.android.volley.Request.Method}
     * @param url  URL of the request to make
     * @param body  body of the request
     * @param clazz  Relevant class object, for Gson's reflection
     * @param headers  Map of request headers
     * @param listener  Listener if request is successful
     * @param errorListener  Listener if request fails
     */
    public GsonRequest(int method, String url, Object body, Class<T> clazz, Map<String, String> headers,
                       Response.Listener<T> listener, Response.ErrorListener errorListener) {
        this(method, url,clazz, headers, listener, errorListener);
        // convert from pojo to json
        if (body != null) {
            this.mRequestBody = gson.toJson(body);
        }
    }

    /*
     * getting the body of our request in bytes
     */
    @Override
    public byte[] getBody() {
        try {
            return mRequestBody == null ? null : mRequestBody.getBytes(PROTOCOL_CHARSET);
        } catch(UnsupportedEncodingException e) {
            VolleyLog.wtf("Unsupported Encoding while trying to get the bytes of %s using %s", mRequestBody, PROTOCOL_CHARSET);
            return null;
        }
    }

    @Override
    public Map<String, String> getHeaders() throws AuthFailureError {
        return headers != null ? headers : super.getHeaders();
    }

    @Override
    protected void deliverResponse(T response) {
        listener.onResponse(response);
    }

    /*@Override
    protected VolleyError parseNetworkError(VolleyError volleyError) {
        LOGD(TAG, "Error Status code : " + volleyError.networkResponse.statusCode);
        if (volleyError.networkResponse != null && volleyError.networkResponse.data != null) {
            LOGD(TAG, "ErrorData : " + new String(volleyError.networkResponse.data));
            VolleyError error = new VolleyError(new String(volleyError.networkResponse.data));
            volleyError = error;
        }
        return volleyError;
    }*/

    @Override
    protected Response<T> parseNetworkResponse(NetworkResponse response) {
        LOGD(TAG, "Status code : " + response.statusCode);
        try {
            String json = new String(
                    response.data,
                    HttpHeaderParser.parseCharset(response.headers));
            LOGD(TAG, "Response : " + json);
            WsHeader wsHeader = (WsHeader) gson.fromJson(json, clazz);
            if (wsHeader != null) {
                if (response.headers.get("eTag") != null) {
                    wsHeader.setEtag(response.headers.get("eTag"));
                }
                wsHeader.setStatusCode(response.statusCode);
                wsHeader.setCalledFromGpsOrBeacon(mCallFromGpsOrBeacon);
                if (response.headers.get("Static-Server-Url") != null) {
                    wsHeader.setServerUrl(response.headers.get("Static-Server-Url"));
                }
            }
            return Response.success(
                    (T) wsHeader,
                    HttpHeaderParser.parseCacheHeaders(response));
        } catch (UnsupportedEncodingException e) {
            return Response.error(new ParseError(response));
        } catch (JsonSyntaxException e) {
            return Response.error(new ParseError(response));
        } catch (NullPointerException e) {
            return Response.error(new ParseError(response));
        }
    }
}
