package com.proximities.sdk;

import android.Manifest;
import android.app.Application;
import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.net.Uri;
import android.nfc.NfcAdapter;
import android.os.Build;
import android.support.v4.content.ContextCompat;

import com.google.android.gms.maps.model.LatLng;
import com.proximities.sdk.activities.PermissionDialogActivity;
import com.proximities.sdk.bridge.OnAccessLocationListener;
import com.proximities.sdk.bridge.OnCampaignURLClickListener;
import com.proximities.sdk.bridge.OnCatchIdentifierListener;
import com.proximities.sdk.bridge.OnCloseCampaignListener;
import com.proximities.sdk.bridge.OnCustomMenuCampaignListener;
import com.proximities.sdk.bridge.OnDetectMockLocationsListener;
import com.proximities.sdk.bridge.OnLocationChangeListener;
import com.proximities.sdk.bridge.OnOpenCampaignListener;
import com.proximities.sdk.bridge.OnRegisterIdentifierListener;
import com.proximities.sdk.bridge.OnResponseFromQrScanListener;
import com.proximities.sdk.bridge.OnRetrievePartnersListener;
import com.proximities.sdk.bridge.OnScannedBeaconsListener;
import com.proximities.sdk.bridge.OnSweepinConnectServiceReady;
import com.proximities.sdk.corekit.CoreKitInstance;
import com.proximities.sdk.interfaces.EnableLocAndBeaconInterface;
import com.proximities.sdk.interfaces.OnGetCampaignsByUserActionListener;
import com.proximities.sdk.json.model.transmitter.Campaign;
import com.proximities.sdk.request.api.CampaignsByUserActionRequest;
import com.proximities.sdk.request.api.FeedRequest;
import com.proximities.sdk.request.api.RegisterUserRequest;
import com.proximities.sdk.util.ProximitiesPrefs;
import com.proximities.sdk.util.SCSdkNotInitializeException;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

public class ProximitiesConfig {

    public static final String TYPE_ALL = "";
    public static final String TYPE_INFORMATION = "information";
    public static final String TYPE_PROMOTION = "promotion";
    public static final String TYPE_EVENT = "event";
    public static final String TYPE_CULTURE = "culture";

    public static final String RICH_CAMPAIGN                 = "rich-notification";
    public static final String WEB_CAMPAIGN                  = "webview-notification";
    public static final String SIMPLE_CAMPAIGN               = "simple-notification";
    public static final String AUDIO_CAMPAIGN                = "audio-notification";
    public static final String LOYALTY_CAMPAIGN              = "loyalty-notification";
    public static final String FULLSCREEN_IMAGE_CAMPAIGN     = "fullscreen-image-notification";

    public static final String USER_ACTION_ALL_CAMPAIGN_RECEIVED            = "anim_received,received_background,received_foreground";
    public static final String USER_ACTION_CAMPAIGN_RECEIVED_IN_FOREGROUND  = "received_foreground";
    public static final String USER_ACTION_CAMPAIGN_RECEIVED_IN_BACKGROUND  = "received_background";
    public static final String USER_ACTION_CAMPAIGN_SAVED                   = "saved";

    protected static final String NOTIFICATION_CHANNEL_ID  = "com.proximities.sdk_channel";

    private static ProximitiesConfig ourInstance = null;

    public static ProximitiesConfig getInstance(){
        if(ourInstance == null)
            ourInstance = new ProximitiesConfig();
        return ourInstance;
    }

    public static List<Campaign> getFavorites() {
        return favorites;
    }

    public static void setFavorites(List<Campaign> favorites) {
        ProximitiesConfig.favorites = favorites;
    }

    private static List<Campaign> favorites;

    public boolean isFavDisable() {
        return isFavDisable;
    }
    public static String getMainActivity(Context ctx) {
        return ProximitiesPrefs.readMainActivity(ctx);
    }
    public String getCurrentActivity() {
        return currentActivity;
    }
    public boolean isUsingCurrentActivity() {
        return isUsingCurrentActivity;
    }
    public boolean isUsingTint() {
        return isUsingTint;
    }
    public boolean isDialogBeforeDeepLinkEnabled() {
        return isDialogBeforeDeepLinkEnabled;
    }
    static boolean isEntryExitLogEnabled() {
        return isEntryExitLogEnabled;
    }

    private static EnableLocAndBeaconInterface mEnableManagersCallback;
    private boolean isFavDisable = false;
    private String currentActivity = "";
    private boolean isUsingCurrentActivity = false;
    private static boolean isEntryExitLogEnabled = false;
    private boolean isDialogBeforeDeepLinkEnabled = true;
    private boolean isUsingTint = true;
    private Context context;

    void setServiceStarted(boolean serviceStarted) {
        isServiceStarted = serviceStarted;
    }

    private boolean isServiceStarted = false;

    /**
     * Initialize the SweepinConnect Sdk. Needs to be called as soon as possible (in the onCreate() of your application class).
     * @param app Your application  class
     */
    public void initSweepinConnectSdk(Application app){
        context = app.getApplicationContext();
        initFeatures();
        if(ProximitiesPrefs.readAnonymousId(app.getApplicationContext()).isEmpty()){
            ProximitiesPrefs.writeAnonymousId(app.getApplicationContext(), UUID.randomUUID().toString());
        }
        CoreKitInstance.initInstance(app);
        PSManager.getInstance().initPSManager(app.getApplicationContext());
    }

    private void initFeatures(){
        ProximitiesPrefs.writeDetectOnlyClosestBeacon(context, false);
        ProximitiesPrefs.writeShareFragment(context, "");
        ProximitiesPrefs.writeForegroundScanPeriod(context, 1100, 0);
        ProximitiesPrefs.writeBackgroundScanPeriod(context, 5000, 25000);
        ProximitiesPrefs.writeEnableSqlite(context, false);
        createChannel();
    }

    private void createChannel(){
        if(android.os.Build.VERSION.SDK_INT >= Build.VERSION_CODES.O){
            NotificationManager mNotificationManager = (NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);
            CharSequence name = context.getString(R.string.app_name);
            int importance = NotificationManager.IMPORTANCE_HIGH;
            NotificationChannel mChannel = new NotificationChannel(NOTIFICATION_CHANNEL_ID, name, importance);
            mChannel.enableLights(true);
            mChannel.setLightColor(com.proximities.sdk.R.color.prxsc_notification_light);
            mChannel.enableVibration(true);
            mChannel.setVibrationPattern(new long[]{500, 500});
            mNotificationManager.createNotificationChannel(mChannel);
        }
    }

    /**
     * Start the sweepin connect service
     * @param sdkReadyCallback
     */
    public void startSweepinConnectService(OnSweepinConnectServiceReady sdkReadyCallback){
        onSweepinConnectServiceReady = sdkReadyCallback;
        if(!isServiceStarted){
            Intent i = new Intent(context, ProximitiesService.class);
            context.getApplicationContext().startService(i);
            isServiceStarted = true;
        }
    }

    public void stopSweepinConnectService(){
        onSweepinConnectServiceReady = null;
        ProximitiesPrefs.writeServiceIsStopped(context, true);
        Intent i = new Intent(context, ProximitiesService.class);
        context.stopService(i);
        isServiceStarted = false;
    }

    static void setManagersCallback(EnableLocAndBeaconInterface callback){
        mEnableManagersCallback = callback;
    }

    /**
     * Ask location permission as from Android M devices
     */
    public void askLocationPermission(OnAccessLocationListener listener){
        onAccessLocationListener = listener;
        if(ContextCompat.checkSelfPermission(context, Manifest.permission.ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED){
            if(CoreKitInstance.getSharedInstance().isAppInForeground()){
                Intent i = new Intent(context, PermissionDialogActivity.class);
                i.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                context.startActivity(i);
            }
        } else {
            if(onAccessLocationListener != null){
                onAccessLocationListener.onAccessLocationGranted();
            }
        }
    }

    /**
     * Start the geofencing system
     */
    public void startLocationManager(){
        try{
            if(mEnableManagersCallback != null){
                if(ContextCompat.checkSelfPermission(context, Manifest.permission.ACCESS_FINE_LOCATION) == PackageManager.PERMISSION_GRANTED){
                    ProximitiesPrefs.writeStartLocationManager(context, true);
                    mEnableManagersCallback.enableLocationManager();
                }
            } else {
                throw new SCSdkNotInitializeException();
            }
        } catch (SCSdkNotInitializeException e) {
            e.printStackTrace();
        }
    }

    /**
     * Start the beacon manager if you want your app to be able to detect beacons
     */
    public void startBeaconManager(){
        try {
            if(mEnableManagersCallback != null){
                if(android.os.Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR2
                        && ContextCompat.checkSelfPermission(context, Manifest.permission.ACCESS_FINE_LOCATION) == PackageManager.PERMISSION_GRANTED){
                    ProximitiesPrefs.writeStartBeaconManager(context, true);
                    mEnableManagersCallback.enableBeaconManager();
                }
            } else {
                throw new SCSdkNotInitializeException();
            }
        } catch (SCSdkNotInitializeException e) {
            e.printStackTrace();
        }
    }

    /**
     * Set the main activity of the application which means this activity
     * will be brought to front when a campaign is closed after reception and selection among notifications (app killed only).
     * @param mainActivity main activity of the application
     */
    public void setMainActivity(Class mainActivity){
        ProximitiesPrefs.writeMainActivity(context, mainActivity.getCanonicalName());
    }

    /**
     * Saves the activity the user was on before receiving a campaign
     * @param currentActivity
     */
    void saveCurrentActivity(Class currentActivity){
        isUsingCurrentActivity = true;
        this.currentActivity = currentActivity.getCanonicalName();
    }

    /**
     * Get the last known location of the user.
     * @return last known location
     */
    public LatLng getLastKnownLocation(){
        return new LatLng(ProximitiesPrefs.readCurrentLat(context), ProximitiesPrefs.readCurrentLng(context));
    }

    /**
     * Start a QR Code scanner to scan Sweepin Qr Code or to open a web url
     */
    public void startQrCodeReader(){
        Intent intent = new Intent(context, ScannerQrActivity.class);
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        context.startActivity(intent);
    }

    /**
     * Open the campaign linked with the detected NFC tag.
     * Use this method in the activity with the NFC intent-filter and the correct scheme/host combinaison.
     * @param intent the intent you receive in your activity
     */
    public void readNfc(Intent intent){
        if (intent.getAction() != null && intent.getAction().equals(NfcAdapter.ACTION_NDEF_DISCOVERED)) {
            ProximitiesNfcManager task = new ProximitiesNfcManager(context, intent.getData());
        }
    }

    public void readQrCode(String content){
        if(content != null && !content.isEmpty()) {
            ProximitiesQrCodeManager qrCodeManager = new ProximitiesQrCodeManager(context, Uri.parse(content));
        }
    }

    /**
     * Enable or disable the display of campaigns at their reception.
     * @param disable true if you don't want campaigns to be displayed, false to re-enable
     */
    public void disableAllCampaigns(boolean disable){
        ProximitiesPrefs.writeDisableAnim(context, disable);
    }

    /**
     * If enabled, a dialog is displayed to the user when a simple notification is received.
     * He can choose to close this dialog if he is not interested in which case the deep link will be ignored.
     * @param isEnabled
     */
    public void enableDialogOnSimpleNotification(boolean isEnabled){
        isDialogBeforeDeepLinkEnabled = isEnabled;
    }

    /**
     * Register the user to the SweepinConnect system to be able to send segmented campaigns on a specific group of users.
     * You can add any parameters and use it later for segmentation.
     * You need to use one of your parameters as identifier, the email is highly recommended.
     *
     *   Example :
     *   Map<String, String> parameters = new HashMap<>();
     *   parameters.put("email", "test@test.fr");
     *   parameters.put("age", 30);
     *   ..
     *   ProximitiesConfig.getInstance().registerIdentifier("test@test.fr", parameters, this);
     */
    public void registerIdentifier(String identifier, Map<String, String> parameters, OnRegisterIdentifierListener listener){
        try {
            Map<String, Object> body = new HashMap<>();
            body.put("identifier", identifier);
            body.put("parameters", parameters);
            ProximitiesPrefs.writeUserEmail(context, identifier);
            RegisterUserRequest userReq = new RegisterUserRequest(context, listener);
            userReq.sendRegisterUser(body);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void removeIdentifier(){
        ProximitiesPrefs.writeUserEmail(context, "");
    }

    /**
     *   Add a fragment at the bottom of rich campaigns allowing users to share the content
     *   @param shareFragment fragment to display
     */
    public void setShareFragment(Class shareFragment){
        ProximitiesPrefs.writeShareFragment(context, shareFragment.getCanonicalName());
    }

    /**
     * You can choose to display campaigns without the favorite item in the toolbar menu
     * @param disableFav true to disable, false if not
     */
    public void disableFavorites(boolean disableFav){
        isFavDisable = disableFav;
    }

    /**
     * @param context
     * @param email
     */
    public void setUserEmail(Context context, String email){
        ProximitiesPrefs.writeUserEmail(context, email);
    }

    /**
     * @param context
     */
    public static String getAnonymousId(Context context){
        return ProximitiesPrefs.readAnonymousId(context);
    }

    /**
     * Request an update of the user's location
     */
    public void requestLocationUpdate(){
        if(ProximitiesPrefs.readStartLocationManager(context)) {
            ProximitiesLocationManager.getInstance().askForManualRequest();
        }
    }

    /**
     * Use the title's font color of a campaign to tint on the favorites icons
     *
     * @param useTint (default = true) tint icons
     */
    public void setTintOnFavoritesIcons(boolean useTint){
        this.isUsingTint = useTint;
    }

    /**
     * Set the location priority for the location request.
     * @param priority default is LocationRequest.PRIORITY_BALANCED_POWER_ACCURACY
     */
    public void setLocationPriority(int priority){
        if(ProximitiesPrefs.readStartLocationManager(context)){
            ProximitiesLocationManager.getInstance().askForPriorityChange(priority);
        }
    }

    public void enableEntryExitLog(boolean enable) {
        isEntryExitLogEnabled = enable;
    }

    /**
     * Get active partners with available campaigns.
     * @param nbElements The 'nbElements' closest active partners from the user's location (0 to get them all)
     * @param campaignType The type of campaign you want to retrieve
     *                      ProximitiesConfig.TYPE_ALL
     *                      ProximitiesConfig.TYPE_INFORMATION
     *                      ProximitiesConfig.TYPE_PROMOTION
     *                      ProximitiesConfig.TYPE_EVENT
     *                      ProximitiesConfig.TYPE_CULTURE
     */
    public void getPartners(int nbElements, String campaignType, OnRetrievePartnersListener callback){
        String pagination = "";
        if(nbElements > 0) {
            pagination = "&page=" + 1 + "&size=" + nbElements;
        }
        FeedRequest feedRequest = new FeedRequest(context, callback);
        feedRequest.executeGet("/" + ProximitiesPrefs.readCurrentLat(context) + "/"
                + ProximitiesPrefs.readCurrentLng(context) + "?type=" + campaignType + pagination);
    }

    /**
     * Request a list of campaigns based on a user action
     * @param userAction ProximitiesConfig.USER_ACTION_CAMPAIGN_RECEIVED or ProximitiesConfig.USER_ACTION_CAMPAIGN_SAVED
     * @param callback callback to receive the list of campaigns
     */
    public void getCampaignsByUserAction(String userAction, OnGetCampaignsByUserActionListener callback){
        CampaignsByUserActionRequest request = new CampaignsByUserActionRequest(context, callback);
        request.requestCampaignsByUserAction(userAction);
    }

    /**
     * Display the campaign in a new activity.
     * @param campaign the campaign you want to display
     */
    public void openCampaign(Campaign campaign){
        PSManager.getInstance().startCampaignDisplayManually(campaign);
    }

    /**
     * The beacon scanner detect all beacons in its range but only consider the closest one.
     * It is particularly useful if you plan to have a route of beacons and the order for receiving the information is important.
     * The beacon configuration is extremely important for the effectiveness of this method.
     * @param isEnabled true if you want to activate this mode
     */
    public void enableClosestBeaconMode(boolean isEnabled){
        ProximitiesPrefs.writeDetectOnlyClosestBeacon(context, isEnabled);
    }

    /**
     * Set duration variables for the beacon scanner when app is in foreground
     * @param foregroundScanPeriod duration of a scan
     * @param foregroundBetweenScanPeriod duration between two scans
     */
    public void setForegroundScanPeriod(int foregroundScanPeriod, int foregroundBetweenScanPeriod){
        ProximitiesPrefs.writeForegroundScanPeriod(context, foregroundScanPeriod, foregroundBetweenScanPeriod);
    }

    /**
     * Set duration variables for the beacon scanner when app is not in foreground (background or killed)
     * @param backgroundScanPeriod duration of a scan
     * @param backgroundBetweenScanPeriod duration between two scans
     */
    public void setBackgroundScanPeriod(int backgroundScanPeriod, int backgroundBetweenScanPeriod){
        ProximitiesPrefs.writeBackgroundScanPeriod(context, backgroundScanPeriod, backgroundBetweenScanPeriod);
    }

    /**
     * Add your UUIDs to use your own beacons
     *
     * @param listOfUuids
     */
    public void addUuidsToBeaconScanner(List<String> listOfUuids){
        if(listOfUuids != null && !listOfUuids.isEmpty()) ProximitiesPrefs.writeUuidsList(context, listOfUuids);
    }

    /**
     * A Sqlite database is created by default to store campaign's logs when the network is unavailable
     * and to store campaigns when a user is near a point of interest where BackgroundFetch is activated.
     * If, for example, the network availability is not an issue, you can choose to disable the creation of this database.
     * (default = True)
     * @param enable
     */
    public void enableSQLiteDatabase(boolean enable){
        ProximitiesPrefs.writeEnableSqlite(context, enable);
    }

    //// * Listeners * ////

    public static OnAccessLocationListener getOnAccessLocationListener() {
        return onAccessLocationListener;
    }

    public static OnCatchIdentifierListener getOnCatchIdentifierListener() {
        return onCatchIdentifierListener;
    }

    public static OnCustomMenuCampaignListener getOnCustomMenuCampaignListener() {
        return onCustomMenuCampaignListener;
    }

    static OnScannedBeaconsListener getOnScannedBeaconsListener() {
        return onScannedBeaconsListener;
    }

    static OnLocationChangeListener getOnLocationChangeListener() {
        return onLocationChangeListener;
    }

    static OnResponseFromQrScanListener getOnResponseFromQrScanListener() {
        return onResponseFromQrScanListener;
    }

    public static OnCampaignURLClickListener getOnCampaignURLClickListener() {
        return onCampaignURLClickListener;
    }

    static OnOpenCampaignListener getOnOpenCampaignListener() {
        return onOpenCampaignListener;
    }

    static OnDetectMockLocationsListener getOnDetectMockLocationsListener() {
        return onDetectMockLocationsListener;
    }

    static OnSweepinConnectServiceReady getOnSweepinConnectServiceReady() {
        return onSweepinConnectServiceReady;
    }

    static OnCloseCampaignListener getOnCloseCampaignListener(){
        return onCloseCampaignListener;
    }

    private static OnSweepinConnectServiceReady onSweepinConnectServiceReady;
    private static OnLocationChangeListener onLocationChangeListener;
    private static OnCatchIdentifierListener onCatchIdentifierListener;
    private static OnAccessLocationListener onAccessLocationListener;
    private static OnCustomMenuCampaignListener onCustomMenuCampaignListener;
    private static OnCampaignURLClickListener onCampaignURLClickListener;
    private static OnScannedBeaconsListener onScannedBeaconsListener;
    private static OnResponseFromQrScanListener onResponseFromQrScanListener;
    private static OnOpenCampaignListener onOpenCampaignListener;
    private static OnDetectMockLocationsListener onDetectMockLocationsListener;
    private static OnCloseCampaignListener onCloseCampaignListener;


    public static void setOnLocationChangeListener(OnLocationChangeListener locationChangeListener){
        onLocationChangeListener = locationChangeListener;
    }

    /**
     * Callback to retrieve the identifier of a simple notification.
     */
    public static void setOnCatchIdentifierListener(OnCatchIdentifierListener catchIdentifierListener){
        onCatchIdentifierListener = catchIdentifierListener;
    }

    /**
     * Callback after the user's response to the location permission. (> API 23)
     public static void setOnAccessLocationListener(OnAccessLocationListener accessLocationListener){
     onAccessLocationListener = accessLocationListener;
     }*/

    /**
     * Callback to replace the favorite menu item by your own menu on all campaigns.
     */
    public static void addOnCustomMenuCampaignListener(OnCustomMenuCampaignListener customMenuCampaignListener){
        onCustomMenuCampaignListener = customMenuCampaignListener;
    }

    public static void setOnCampaignURLClickListener(OnCampaignURLClickListener listener){
        onCampaignURLClickListener = listener;
    }

    /**
     * Callback to retrieve the list of scanned beacons.
     */
    public static void setOnScannedBeaconsListener(OnScannedBeaconsListener scannedBeaconsListener){
        onScannedBeaconsListener = scannedBeaconsListener;
    }

    /**
     * Callback to handle specific situations occuring after a Qr Code scan.
     */
    public static void setOnResponseFromQrScanListener(OnResponseFromQrScanListener responseFromQrScanListener){
        onResponseFromQrScanListener = responseFromQrScanListener;
    }

    /**
    * Callback to add actions when a campaign is received and opened
    */
    public static void setOnOpenCampaignListener(OnOpenCampaignListener openCampaignListener){
        onOpenCampaignListener = openCampaignListener;
    }

    /**
     * Callback to detect if the lastest location update is from a mock provider
     */
    public static void setOnDetectMockLocationsListener(OnDetectMockLocationsListener detectMockLocationsListener){
        onDetectMockLocationsListener = detectMockLocationsListener;
    }

    /**
     * Callback when a received campaign is closed by the user
     * @param closeCampaignListener
     */
    public static void setOnCloseCampaignListener(OnCloseCampaignListener closeCampaignListener){
        onCloseCampaignListener = closeCampaignListener;
    }
}

