package com.proximities.sdk;

import android.app.Activity;
import android.app.Notification;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.drawable.Drawable;
import android.media.RingtoneManager;
import android.os.Build;
import android.support.v4.app.NotificationCompat;
import android.support.v4.content.ContextCompat;
import android.widget.RemoteViews;

import com.bumptech.glide.Glide;
import com.bumptech.glide.request.animation.GlideAnimation;
import com.bumptech.glide.request.target.SimpleTarget;
import com.proximities.sdk.corekit.CoreKitInstance;
import com.proximities.sdk.corekit.utils.CKUtils;
import com.proximities.sdk.activities.CampaignActivity;
import com.proximities.sdk.activities.EmptyCampaignActivity;
import com.proximities.sdk.activities.LoyaltyCampaignActivity;
import com.proximities.sdk.activities.MultiCampaignsActivity;
import com.proximities.sdk.activities.SimpleNotificationActivity;
import com.proximities.sdk.activities.WebViewActivity;
import com.proximities.sdk.json.model.transmitter.Campaign;
import com.proximities.sdk.util.ProximitiesConstants;
import com.proximities.sdk.util.ProximitiesPrefs;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by william on 08/12/16.
 */

/*
    Handle display of campaigns (background and foreground)
 */
class PSManager{

    private static final PSManager INSTANCE = new PSManager();

    private PSManager(){}

    public static PSManager getInstance() {
        return INSTANCE;
    }

    private Context context;
    private int idCurrentNotification = 0;
    private boolean isCampaignCurrentlyOpen;
    private boolean isLoyaltyCampaignOpen;
    private LogsManager mLogsManager;
    private List<String> campaignClasses; // activities concerning the display of campaigns
    private List<Campaign> displayedCampaigns; // campaigns currently displayed

    void initPSManager(Context context){
        this.context = context;
        this.displayedCampaigns = new ArrayList<>();
        if(ProximitiesPrefs.readEnableSqlite(context)){
            DatabaseManager.getInstance().initDatabaseManager(context);
        }
        mLogsManager = LogsManager.getInstance();
        mLogsManager.initLogsManager(context);
        isCampaignCurrentlyOpen = false;
        initCampaignClassesNames();
    }

    /**
     * Identify the last resumed activity has a 'campaign' activity or not
     * @param activity that has just been resumed
     */
    void handleCampaignsActivities(Activity activity){
        boolean isCampaignActivity = false;
        for (String className : campaignClasses){
            if(activity.getClass().getCanonicalName().equals(className)){
                isCampaignActivity = true;
                break;
            }
        }
        if(isCampaignActivity){
            isCampaignCurrentlyOpen = true;
            isLoyaltyCampaignOpen = activity.getClass().getCanonicalName().equals(LoyaltyCampaignActivity.class.getCanonicalName());
        } else {
            ProximitiesConfig.getInstance().saveCurrentActivity(activity.getClass());
            displayedCampaigns.clear();
            isCampaignCurrentlyOpen = false;
            isLoyaltyCampaignOpen = false;
        }
    }

    void removeCampaignFromDisplayedList(Campaign campaign){
        displayedCampaigns.remove(campaign);
    }

    /**
     * Display the campaigns received via beacons or geofence if no campaign are currently displayed
     * @param campaigns is the list of newly received campaigns
     * @param poiId
     * @param transmitterId
     */
    void startCampaignsDisplay(List<Campaign> campaigns, String poiId, String transmitterId){
        if(!isCampaignCurrentlyOpen || isLoyaltyRefreshed(campaigns)) {
            handleCampaignsDisplay(campaigns, poiId, transmitterId);
        }
    }

    void startCampaignsDisplayFromNfc(List<Campaign> campaigns, String poiId, String transmitterId){
        handleCampaignsDisplay(campaigns, poiId, transmitterId);
    }

    void startCampaignsDisplayFromQRCode(List<Campaign> campaigns, String poiId, String transmitterId){
        handleCampaignsDisplay(campaigns, poiId, transmitterId);
    }

    void startCampaignDisplayManually(Campaign campaign){
        ArrayList<Campaign> campaigns = new ArrayList<>();
        campaigns.add(campaign);
        addCampaignsToDisplayList(campaigns);
        Intent intent;
        switch(campaign.getTemplateType()){
            case ProximitiesConfig.SIMPLE_CAMPAIGN:
                intent = new Intent(context, SimpleNotificationActivity.class);
                break;
            case ProximitiesConfig.LOYALTY_CAMPAIGN:
                intent = new Intent(context, LoyaltyCampaignActivity.class);
                break;
            default:
                intent = new Intent(context, CampaignActivity.class);
                break;
        }
        intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK | Intent.FLAG_ACTIVITY_REORDER_TO_FRONT);
        intent.putExtra(ProximitiesConstants.LIST_CAMPAIGNS_EXTRA, (ArrayList) displayedCampaigns);
        context.startActivity(intent);
    }

    private void handleCampaignsDisplay(List<Campaign> campaigns, String poiId, String transmitterId){
        addCampaignsToDisplayList(campaigns);
        Intent intent = null;
        if (displayedCampaigns.size() == 1) {
            switch(displayedCampaigns.get(0).getTemplateType()){
                case ProximitiesConfig.SIMPLE_CAMPAIGN:
                    intent = new Intent(context, SimpleNotificationActivity.class);
                    break;
                case ProximitiesConfig.LOYALTY_CAMPAIGN:
                    intent = new Intent(context, LoyaltyCampaignActivity.class);
                    break;
                default:
                    intent = new Intent(context, CampaignActivity.class);
                    break;
            }
        } else if (displayedCampaigns.size() > 1) {
            intent = new Intent(context, MultiCampaignsActivity.class);
            intent.putExtra(ProximitiesConstants.IS_MULTI_EXTRA, true);
        }

        if (intent != null){
            intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK | Intent.FLAG_ACTIVITY_REORDER_TO_FRONT);
            intent.putExtra(ProximitiesConstants.LIST_CAMPAIGNS_EXTRA, (ArrayList) displayedCampaigns);
            intent.putExtra(ProximitiesConstants.CAMPAIGN_RECEIVED_EXTRA, true);
            if(poiId != null) {
                intent.putExtra(ProximitiesConstants.POI_ID_EXTRA, poiId);
            } else if (transmitterId != null) {
                intent.putExtra(ProximitiesConstants.TRANSMITTER_ID_EXTRA, transmitterId);
            }
            if(!CoreKitInstance.getSharedInstance().isAppInForeground()){
                mLogsManager.generateLogs(displayedCampaigns, ProximitiesConstants.LOG_ACTION_RECEIVED_BACKGROUND,  poiId, transmitterId);
                intent.putExtra(ProximitiesConstants.ANIM_BY_PUSH_EXTRA, true);
                if(displayedCampaigns.size() == 1) {
                    generatePushForCampaign(intent, displayedCampaigns.get(0));
                } else {
                    generatePushForMultiCampaigns(intent);
                }
            } else {
                mLogsManager.generateLogs(displayedCampaigns, ProximitiesConstants.LOG_ACTION_RECEIVED_FOREGROUND,  poiId, transmitterId);
                context.startActivity(intent);
            }
        }
    }

    /**
     * Update the list of campaigns currently displayed with the new campaigns received
     * @param campaigns is the list of newly received campaigns
     */
    private void addCampaignsToDisplayList(List<Campaign> campaigns){
        if(displayedCampaigns.size() == 0){
            displayedCampaigns.addAll(campaigns);
        }else{
            for(Campaign received : campaigns){
                for(Campaign displayed : displayedCampaigns){
                    if(received.getId() == displayed.getId()){
                        displayedCampaigns.remove(displayed);
                        break;
                    }
                }
                displayedCampaigns.add(received);
            }
        }
    }

    private void generatePushForMultiCampaigns(Intent intent){
        createNotification(context.getString(R.string.prxsc_multi_campaigns_notification_content), intent, null, null);
    }

    /**
     * Generate a push or let user to generate push
     *
     * @param intent {@link android.content.Intent} to run when user clicks on notif
     * @param anim {@link Campaign} or null if there are multiple available animation
     */
    private void generatePushForCampaign(final Intent intent, final Campaign anim) {
        final RemoteViews contentView = new RemoteViews(context.getPackageName(), R.layout.prxsc_notification_layout);

        if(anim.getPoi() != null && !anim.getPoi().getName().isEmpty()){
            contentView.setTextViewText(R.id.poi_name, anim.getPoi().getName());
        }

        if(anim.getTitle() != null && !anim.getTitle().isEmpty()){
            contentView.setTextViewText(R.id.content, CKUtils.fromHtml(anim.getTitle()));
        }

        contentView.setImageViewResource(R.id.logo, R.drawable.prxsc_ic_notification);
        if(anim.getImage() != null && !anim.getImage().isEmpty()) {
            Glide.with(context).load(ProximitiesConstants.STATIC_CONTENT_HOST + anim.getImage())
                    .asBitmap()
                    .into(new SimpleTarget<Bitmap>() {
                        @Override
                        public void onResourceReady(Bitmap resource, GlideAnimation glideAnimation) {
                            contentView.setImageViewBitmap(R.id.image, resource);
                            createNotification(CKUtils.fromHtml(anim.getTitle()), intent, contentView, resource);
                        }

                        @Override
                        public void onLoadFailed(Exception e, Drawable errorDrawable) {
                            createNotification(CKUtils.fromHtml(anim.getTitle()), intent, null, null);
                            super.onLoadFailed(e, errorDrawable);
                        }
                    });
        } else {
            createNotification(CKUtils.fromHtml(anim.getTitle()), intent, null, null);
        }
    }

    private void createNotification(String msg, Intent intent, RemoteViews contentView, Bitmap bitmap) {
        if(android.os.Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            NotificationCompat.Builder mBuilder = new NotificationCompat.Builder(context, ProximitiesConfig.NOTIFICATION_CHANNEL_ID)
                    .setSmallIcon(R.drawable.prxsc_ic_small_notification)
                    .setLargeIcon((bitmap == null) ? (BitmapFactory.decodeResource(context.getResources(), R.drawable.prxsc_ic_notification)) : bitmap)
                    .setContentTitle(context.getString(R.string.app_name))
                    .setContentText(msg)
                    .setVibrate(new long[]{500, 500})
                    .setSound(RingtoneManager.getDefaultUri(RingtoneManager.TYPE_NOTIFICATION))
                    .setAutoCancel(true)
                    .setLights(ContextCompat.getColor(context, com.proximities.sdk.R.color.prxsc_notification_light), 1000, 2500);

            if (contentView != null) {
                mBuilder.setCustomContentView(contentView);
            }

            PendingIntent resultPendingIntent = PendingIntent.getActivity(context, idCurrentNotification, intent, PendingIntent.FLAG_UPDATE_CURRENT);
            mBuilder.setContentIntent(resultPendingIntent);

            NotificationManager mNotificationManager = (NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);
            mNotificationManager.notify(idCurrentNotification, mBuilder.build());

        } else {
            Notification notif = new Notification.Builder(context)
                    .setSmallIcon(R.drawable.prxsc_ic_small_notification)
                    .setLargeIcon((bitmap == null) ? (BitmapFactory.decodeResource(context.getResources(), R.drawable.prxsc_ic_notification)) : bitmap)
                    .setPriority(Notification.PRIORITY_HIGH)
                    .setContentTitle(context.getString(R.string.app_name))
                    .setContentText(msg)
                    .build();

            if(contentView != null){
                notif.bigContentView = contentView;
            }
            notif.contentIntent = PendingIntent.getActivity(context, idCurrentNotification, intent, PendingIntent.FLAG_UPDATE_CURRENT);
            notif.vibrate = new long[]{500, 500};
            notif.ledARGB = ContextCompat.getColor(context, R.color.prxsc_notification_light);
            notif.flags = Notification.FLAG_SHOW_LIGHTS;
            notif.flags = Notification.FLAG_AUTO_CANCEL;
            notif.ledOnMS = 1000;
            notif.ledOffMS = 2500;

            NotificationManager mNotifyManager = (NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);
            mNotifyManager.notify(idCurrentNotification, notif);
        }

        idCurrentNotification++;
        displayedCampaigns.clear();
    }

    /*
     *  All the single task classes concerning campaign display
     *  -> Used to determine if a campaign is open
     */
    private void initCampaignClassesNames(){
        campaignClasses = new ArrayList<>();
        campaignClasses.add(CampaignActivity.class.getCanonicalName());
        campaignClasses.add(LoyaltyCampaignActivity.class.getCanonicalName());
        campaignClasses.add(MultiCampaignsActivity.class.getCanonicalName());
        campaignClasses.add(SimpleNotificationActivity.class.getCanonicalName());
        campaignClasses.add(EmptyCampaignActivity.class.getCanonicalName());
        campaignClasses.add(WebViewActivity.class.getCanonicalName());
    }

    /**
     * @param campaigns is the list of newly received campaigns
     * @return true if a loyalty campaign is open and if the new received campaign is the same loyalty
     */
    private boolean isLoyaltyRefreshed(List<Campaign> campaigns){
        return (isLoyaltyCampaignOpen && campaigns.size() == 1 && campaigns.get(0).getId() == displayedCampaigns.get(0).getId());
    }

}
