package com.proximities.sdk.corekit;

import android.Manifest;
import android.content.Context;
import android.content.pm.PackageManager;
import android.location.Location;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.v4.content.ContextCompat;

import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.api.GoogleApiClient;
import com.google.android.gms.location.LocationListener;
import com.google.android.gms.location.LocationRequest;
import com.google.android.gms.location.LocationServices;
import com.proximities.sdk.corekit.interfaces.IPSLocationServiceInterface;
import com.proximities.sdk.corekit.interfaces.PSLocationServiceInterface;

/**
 * Created by william on 05/12/16.
 */

public class CKLocationManager implements GoogleApiClient.ConnectionCallbacks, GoogleApiClient.OnConnectionFailedListener, LocationListener{

    private static CKLocationManager sharedInstance;

    public static CKLocationManager getSharedInstance(Context context){
        if(sharedInstance == null){
            sharedInstance = new CKLocationManager(context);
        }
        return sharedInstance;
    }

    private Context context;
    private GoogleApiClient googleApiClient;
    private LocationRequest locationRequest;
    private PSLocationServiceInterface mPSCallback;
    private IPSLocationServiceInterface mIPSCallback;
    private boolean isRequestedManually = false;

    public CKLocationManager(Context context){
        if (googleApiClient == null) {
            this.context = context;
            googleApiClient = new GoogleApiClient.Builder(context)
                    .addApi(LocationServices.API)
                    .addConnectionCallbacks(this)
                    .addOnConnectionFailedListener(this)
                    .build();
        }
    }

    public void connectGoogleApiClient(){
        if(!isConnected()){
            googleApiClient.connect();
        }
    }

    public void createLocationRequest(int priority, float smallestDisplacement, long interval, long fastestInterval){
        if(locationRequest == null){
            locationRequest = LocationRequest.create();
            locationRequest.setPriority(priority);
            locationRequest.setInterval(interval);
            locationRequest.setFastestInterval(fastestInterval);
            locationRequest.setSmallestDisplacement(smallestDisplacement);
        }
    }

    public void updateLocationRequest(float smallestDisplacement, long interval, long fastestInterval){
        if(locationRequest != null) {
            locationRequest.setInterval(interval);
            locationRequest.setFastestInterval(fastestInterval);
            locationRequest.setSmallestDisplacement(smallestDisplacement);
        }
    }

    public void setLocationPriority(int priority){
        if(locationRequest != null){
            locationRequest.setPriority(priority);
            askForLocationUpdate();
        }
    }

    public void requestLocationUpdate(){
        if(ContextCompat.checkSelfPermission(context, Manifest.permission.ACCESS_FINE_LOCATION) == PackageManager.PERMISSION_GRANTED && isConnected()){
            LocationServices.FusedLocationApi.requestLocationUpdates(googleApiClient, locationRequest, this);
        }
    }

    public void removeLocationUpdate(){
        LocationServices.FusedLocationApi.removeLocationUpdates(googleApiClient, this);
    }

    public void askForLocationUpdate() {
        if (isConnected()) {
            isRequestedManually = true;
            removeLocationUpdate();
            requestLocationUpdate();
        }
    }

    public void resetManualRequest() {
        isRequestedManually = false;
    }

    @Override
    public void onConnected(@Nullable Bundle bundle) {
        if(mPSCallback != null){
            mPSCallback.onConnected();
        }
        if(mIPSCallback != null){
            mIPSCallback.onConnected();
        }
    }

    @Override
    public void onLocationChanged(Location location) {
        if(mPSCallback != null){
            mPSCallback.onLocationChanged(location, isRequestedManually);
        }
        if(mIPSCallback != null){
            mIPSCallback.onLocationChanged(location, isRequestedManually);
        }
    }

    @Override
    public void onConnectionSuspended(int i) {

    }

    @Override
    public void onConnectionFailed(@NonNull ConnectionResult connectionResult) {

    }

    private boolean isConnected() {
        return googleApiClient != null && googleApiClient.isConnected();
    }

    public void attachPSLocationInterface(PSLocationServiceInterface callback){
        this.mPSCallback = callback;
    }

    public void attachIPSLocationInterface(IPSLocationServiceInterface callback){
        this.mIPSCallback = callback;
    }
}
