package com.proximities.sdk;

import android.content.Context;

import com.proximities.sdk.interfaces.LogDatabaseInterface;
import com.proximities.sdk.interfaces.LogInterface;
import com.proximities.sdk.json.model.log.BaseLogResponse;
import com.proximities.sdk.json.model.log.CampaignLogs;
import com.proximities.sdk.json.model.log.Log;
import com.proximities.sdk.json.model.partner.Poi;
import com.proximities.sdk.json.model.transmitter.Campaign;
import com.proximities.sdk.json.model.transmitter.Transmitter;
import com.proximities.sdk.request.api.LogDatabaseRequest;
import com.proximities.sdk.request.api.LogRequest;
import com.proximities.sdk.util.ProximitiesConstants;
import com.proximities.sdk.util.ProximitiesPrefs;
import com.proximities.sdk.util.Utils;

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.List;

/**
 * Created by william on 13/12/16.
 */

public class LogsManager implements LogInterface, LogDatabaseInterface{

    private static final LogsManager INSTANCE = new LogsManager();

    private DatabaseManager mDbAccessor;
    private LogRequest mLogRequest;
    private LogDatabaseRequest mLogDbRequest;
    private boolean isSqliteEnabled = true;

    public static LogsManager getInstance() {
        return INSTANCE;
    }

    private LogsManager(){}

    void initLogsManager(Context context) {
        isSqliteEnabled = ProximitiesPrefs.readEnableSqlite(context);
        mLogRequest = new LogRequest(context, this);
        if(isSqliteEnabled){
            mLogDbRequest = new LogDatabaseRequest(context, this);
            mDbAccessor = DatabaseManager.getInstance();
        }
    }

    /**
     * Create and send logs for each campaign
     * @param campaigns current campaigns
     * @param action action to send for each campaign
     *                   {@link com.proximities.sdk.util.ProximitiesConstants}
     * @param poiId
     * @param transmitterId
     */
    void generateLogs(List<Campaign> campaigns, String action, String poiId, String transmitterId){
        CampaignLogs campaignLogs = new CampaignLogs();
        for (Campaign campaign : campaigns) {
            String poiIdToSend = null;
            if(poiId != null){
                poiIdToSend = (poiId.equals(ProximitiesConstants.ORIGIN_MULTIPLE_POIS)) ? String.valueOf(campaign.getPoi().getId()) : poiId;
            }
            campaignLogs.addLog(new Log(campaign.getId(), action, Utils.getCurrentDate(), poiIdToSend, transmitterId));
        }
        /*if(isSqliteEnabled){
            mDbAccessor.insertLogsIntoDatabase(campaignLogs);
        }*/
        mLogRequest.executePost(campaignLogs);
    }

    void generateLogsWithoutOrigins(List<Campaign> campaigns, String action){
        CampaignLogs campaignLogs = new CampaignLogs();
        for (Campaign campaign : campaigns) {
            campaignLogs.addLog(new Log(campaign.getId(), action, Utils.getCurrentDate()));
        }
        /*if(isSqliteEnabled){
            mDbAccessor.insertLogsIntoDatabase(campaignLogs);
        }*/
        mLogRequest.executePost(campaignLogs);
    }

    void executeDatabaseReset(){
        if(isSqliteEnabled){
            mDbAccessor.executeDatabaseReset(mLogDbRequest);
        }
    }

    void saveCampaignsInDatabase(Poi poi, Transmitter transmitter){
        if(isSqliteEnabled){
            mDbAccessor.saveCampaignInDatabase(poi, transmitter);
        }
    }

    @Override
    public void onSendLogSuccess(BaseLogResponse logs) {
//
    }

    @Override
    public void onSendLogError(CampaignLogs logs) {
        if(isSqliteEnabled){
            mDbAccessor.insertLogsIntoDatabase(logs);
        }
    }

    @Override
    public void onSendLogFromDatabaseSuccess() {
        if(isSqliteEnabled){
            mDbAccessor.deleteLogsFromDatabase();
        }
    }

    @Override
    public void onSendLogFromDatabaseError() {

    }

}
