package com.proximities.sdk;

import android.Manifest;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.os.RemoteException;
import android.support.v4.content.ContextCompat;
import android.util.Log;
import android.widget.Toast;

import com.proximities.sdk.bridge.OnEventCallTransmitters;
import com.proximities.sdk.util.ProximitiesPrefs;

import org.altbeacon.beacon.Beacon;
import org.altbeacon.beacon.BeaconConsumer;
import org.altbeacon.beacon.BeaconManager;
import org.altbeacon.beacon.BeaconParser;
import org.altbeacon.beacon.Identifier;
import org.altbeacon.beacon.MonitorNotifier;
import org.altbeacon.beacon.RangeNotifier;
import org.altbeacon.beacon.Region;
import org.altbeacon.beacon.distance.CurveFittedDistanceCalculator;
import org.altbeacon.beacon.distance.ModelSpecificDistanceCalculator;
import org.altbeacon.beacon.service.ArmaRssiFilter;
import org.altbeacon.beacon.startup.RegionBootstrap;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import static com.proximities.sdk.util.LogUtils.LOGD;
import static com.proximities.sdk.util.LogUtils.LOGE;
import static com.proximities.sdk.util.LogUtils.makeLogTag;

/**
 * Created by Antoine Arnoult <arnoult.antoine@gmail.com> on 27/12/14.
 */
public class ProximitiesBeaconManager /*implements RangeNotifier*/{

    private static final String TAG = makeLogTag(ProximitiesBeaconManager.class);

    private static final ProximitiesBeaconManager ourInstance = new ProximitiesBeaconManager();

    private static final String CUSTOM_IBEACON_LAYOUT = "m:2-3=0215,i:4-19,i:20-21,i:22-23,p:24-24";
    private static final String CUSTOM_EDDYSTONE_LAYOUT = "s:0-1=feaa,m:2-2=00,p:3-3:-41,i:4-13,i:14-19";

    private static final String DEFAULT_UUID_ONE = "F46EFD73-819B-4766-AB45-CAE5A282CD59";
    private static final String DEFAULT_UUID_TWO = "12E4CE63-E9D1-40E9-A529-096DE63E504E";
    private static final String DEFAULT_UUID_THREE = "F0018B9B-7509-4C31-A905-1A27D39C003C";
    private static final String DEFAULT_NAMESPACE_ONE = "EDD1EBEAC04E5DEFA017";

    private static long mForegroundBetweenScanPeriod = BeaconManager.DEFAULT_FOREGROUND_BETWEEN_SCAN_PERIOD;
    private static long mForegroundScanPeriod = BeaconManager.DEFAULT_FOREGROUND_SCAN_PERIOD;
    private static long mBackgroundBetweenScanPeriod = BeaconManager.DEFAULT_FOREGROUND_BETWEEN_SCAN_PERIOD;
    private static long mBackgroundScanPeriod = BeaconManager.DEFAULT_BACKGROUND_SCAN_PERIOD;

    private Context ctx;
    private BeaconManager beaconManager;
    private List<Region> regions;
    private OnEventCallTransmitters callbackTransmitters;
    private boolean hasDetectedNewBeacon = false;

    private long lastWsCallTs;
    private static int MIN_WS_CALL_TS = 1500;

    public static ProximitiesBeaconManager getInstance() {
        return ourInstance;
    }

    private ProximitiesBeaconManager() {}

    public void initBeaconManager(Context ctx, BeaconManager beaconManager, OnEventCallTransmitters callbackTransmitters, BeaconConsumer consumer) {
        this.ctx = ctx;
        this.callbackTransmitters = callbackTransmitters;
        this.beaconManager = beaconManager;
        try {
            // @See http://stackoverflow.com/questions/25027983/is-this-the-correct-layout-to-detect-ibeacons-with-altbeacons-android-beacon-li
            this.beaconManager.getBeaconParsers().add(new BeaconParser().setBeaconLayout(CUSTOM_EDDYSTONE_LAYOUT));
            this.beaconManager.getBeaconParsers().add(new BeaconParser().setBeaconLayout(CUSTOM_IBEACON_LAYOUT));
            this.beaconManager.bind(consumer);
        } catch (UnsupportedOperationException exception) {
            LOGD(TAG, "unmodifiableList");
        }
        initRegions();
        LOGD(TAG, "------> Init beacon manager");
        setForegroundScanPeriod(ProximitiesPrefs.readForegroundScanPeriod(ctx), ProximitiesPrefs.readForegroundBetweenScanPeriod(ctx));
        setBackgroundScanPeriod(ProximitiesPrefs.readBackgroundScanPeriod(ctx), ProximitiesPrefs.readBackgroundBetweenScanPeriod(ctx));
    }

    public BeaconManager getBeaconManager() {
        return beaconManager;
    }

    /**
     * Make a list with default regions which can be
     *
     * {@value #DEFAULT_UUID_ONE}
     * {@value #DEFAULT_UUID_TWO}
     */
    public List<Region> initRegions() {
        if(regions == null) regions = new ArrayList<>();
        regions.clear();
        regions.add(new Region(DEFAULT_UUID_ONE, Identifier.parse(DEFAULT_UUID_ONE), null, null));
        regions.add(new Region(DEFAULT_UUID_TWO, Identifier.parse(DEFAULT_UUID_TWO), null, null));
        regions.add(new Region(DEFAULT_NAMESPACE_ONE, Identifier.parse(DEFAULT_NAMESPACE_ONE), null, null));
        regions.add(new Region(DEFAULT_UUID_THREE, Identifier.parse(DEFAULT_UUID_TWO), null, null));

        if(ProximitiesPrefs.readUuidsList(ctx) != null && ProximitiesPrefs.readUuidsList(ctx).length > 0) {
            for(String uuid : ProximitiesPrefs.readUuidsList(ctx)){
                try {
                    regions.add(new Region(uuid, Identifier.parse(uuid), null, null));
                } catch (IllegalArgumentException exception){
                    LOGE(TAG, "Unable to add the uuid : " + uuid + ". Wrong format");
                }
            }
        }

        return regions;
    }

    public void addRegion(String uuid) {
        if (uuid != null && !uuid.isEmpty()) {
            try {
                Region region = new Region(uuid, Identifier.parse(uuid), null, null);
                regions.add(region);
            } catch(IllegalArgumentException e) {
                LOGE(TAG, "Unable to parse : " + uuid);
            }
        }
    }

    public void startMonitoring() {
        LOGD(TAG, "Start monitoring");
        beaconManager.addMonitorNotifier(new MonitorNotifier() {
            @Override
            public void didEnterRegion(Region region) {
                hasDetectedNewBeacon = true;
                LOGD(TAG, "I just saw a beacon for the first time! ");
//                addRangingRegions();
//                beaconManager.setRangeNotifier(ProximitiesBeaconManager.this);
            }

            @Override
            public void didExitRegion(Region region) {
                if(ProximitiesConfig.isEntryExitLogEnabled()) {
                    callbackTransmitters.onExitLogTransmitter(ProximitiesPrefs.readLastBeaconDetected(ctx));
                    ProximitiesPrefs.writeLastBeaconDetected(ctx, "");
                }
                LOGD(TAG, "I no longer see a beacon");
            }

            @Override
            public void didDetermineStateForRegion(int state, Region region) {
                LOGD(TAG, "I have just switched from seeing/not seeing beacons: " + state);
            }
        });

        addMonitoringRegions();
    }

    /**
     * Add monitoring regions to {@link BeaconManager}
     */
    private void addMonitoringRegions() {
        try {
            for (Region r : regions) {
                beaconManager.startMonitoringBeaconsInRegion(r);
            }
        } catch (RemoteException e) { LOGE(TAG, "Remote exception"); }
    }

    /*@Override
    public void didRangeBeaconsInRegion(Collection<Beacon> beacons, Region region) {
        if (ContextCompat.checkSelfPermission(ctx, Manifest.permission.ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED) {
            Intent i = new Intent(ctx, ProximitiesService.class);
            ctx.stopService(i);
        }

        Beacon closestBeacon = null;
        for (Beacon beacon : beacons) {
            if(closestBeacon == null) closestBeacon = beacon;
            else if(closestBeacon.getDistance() > beacon.getDistance()) closestBeacon = beacon;
        }
        selectClosestBeacon(closestBeacon, beacons);

    }*/

    public void startRanging() {
        LOGD(TAG, "Start ranging");
        beaconManager.addRangeNotifier(new RangeNotifier() {
            @Override
            public void didRangeBeaconsInRegion(Collection<Beacon> beacons, Region region) {
                if (ContextCompat.checkSelfPermission(ctx, Manifest.permission.ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED) {
                    Intent i = new Intent(ctx, ProximitiesService.class);
                    ctx.stopService(i);
                }
                if(ProximitiesConfig.getOnScannedBeaconsListener() != null) ProximitiesConfig.getOnScannedBeaconsListener().onScannedBeacons(beacons);
                /*Beacon closestBeacon = null;
                for (Beacon beacon : beacons) {
                    if(closestBeacon == null) closestBeacon = beacon;
                    else if(closestBeacon.getDistance() > beacon.getDistance()) closestBeacon = beacon;
                }
                if(closestBeacon != null) selectClosestBeacon(closestBeacon, beacons);*/
                for (Beacon beacon : beacons) {
                    selectClosestBeacon(beacon, beacons);
                }
            }
        });
        addRangingRegions();
    }

    private void selectClosestBeacon(Beacon beacon, Collection<Beacon> beacons){
        if (beacon.getServiceUuid() == 0xfeaa) {
            LOGD(TAG, "The first Eddystone I see is about "
                    + beacon.getDistance() + " meters away.");
            String namespace = beacon.getId1().toString().substring(2);
            String instance = beacon.getId2().toString().substring(2);
            callbackTransmitters.onCallTransmitters(null, null, null, namespace, instance);
        } else {
            LOGD(TAG, "The first beacon I see is about " + beacon.getDistance() + " meters away.");
            String uuid = beacon.getId1().toString();
            String major = beacon.getId2().toString();
            String minor = beacon.getId3().toString();
            LOGD(TAG, "id1[uuid] : " + uuid);
            LOGD(TAG, "id2[major] : " + major);
            LOGD(TAG, "id3[minor] : " + minor);
            String beaconLog = uuid + "--" + major + "--" + minor;
            //if(!ProximitiesPrefs.readLastBeaconDetected(ctx).equals(beaconLog)){
            if(ProximitiesConfig.isEntryExitLogEnabled() && beacons.size() == 1) {
                if (hasDetectedNewBeacon && !ProximitiesPrefs.readLastBeaconDetected(ctx).equals(beaconLog)) {
                    if (!ProximitiesPrefs.readLastBeaconDetected(ctx).isEmpty())
                        callbackTransmitters.onExitLogTransmitter(ProximitiesPrefs.readLastBeaconDetected(ctx));
                    ProximitiesPrefs.writeLastBeaconDetected(ctx, beaconLog);
                    callbackTransmitters.onEntryLogTransmitter(beaconLog);
                    hasDetectedNewBeacon = false;
                }
            }
            callbackTransmitters.onCallTransmitters(uuid, major, minor, null, null);
        }
    }

    /**
     * Add ranging regions to {@link BeaconManager}
     */
    private void addRangingRegions() {
        try {
            for (Region r : regions) {
                beaconManager.startRangingBeaconsInRegion(r);
            }
        } catch (RemoteException e) { LOGE(TAG, "Remote exception"); }
    }


    /**
     * Set the duration of the scan in milliseconds and the duration between each Bluetooth LE scan cycle to look for beacons (app in foreground)
     *
     * @param scanPeriodMillis time in milliseconds
     * @param waitTimeMillis time in milliseconds
     */
    private void setForegroundScanPeriod(long scanPeriodMillis, long waitTimeMillis) {
        mForegroundScanPeriod = scanPeriodMillis;
        mForegroundBetweenScanPeriod = waitTimeMillis;
        beaconManager.setForegroundScanPeriod(mForegroundScanPeriod);
        beaconManager.setForegroundBetweenScanPeriod(mForegroundBetweenScanPeriod);
        try {
            beaconManager.updateScanPeriods();
        } catch (RemoteException e) {
            LOGE(TAG, "scan period can't be updated");
        }
    }

    /**
     * Set the duration of the scan in milliseconds and the duration between each Bluetooth LE scan cycle to look for beacons (app in background)
     *
     * @param scanPeriodMillis time in milliseconds
     * @param waitTimeMillis time in milliseconds
     */
    private void setBackgroundScanPeriod(long scanPeriodMillis, long waitTimeMillis) {
        mBackgroundScanPeriod = scanPeriodMillis;
        mBackgroundBetweenScanPeriod = waitTimeMillis;
        beaconManager.setBackgroundScanPeriod(mBackgroundScanPeriod);
        beaconManager.setBackgroundBetweenScanPeriod(mBackgroundBetweenScanPeriod);
        try {
            beaconManager.updateScanPeriods();
        } catch (RemoteException e) {
            LOGE(TAG, "scan period can't be updated");
        }
    }
}
