package com.proximities.sdk.request;

import android.content.Context;
import android.util.Log;

import com.proximities.sdk.util.ProximitiesConstants;
import com.proximities.sdk.util.ProximitiesPrefs;
import com.proximities.sdk.util.Utils;

import java.io.IOException;
import java.net.URISyntaxException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import static com.proximities.sdk.util.LogUtils.LOGD;
import static com.proximities.sdk.util.LogUtils.makeLogTag;

/**
 * Created by Antoine Arnoult <arnoult.antoine@gmail.com> on 07/12/14.
 *
 * Manage all headers for application
 */
public final class Header {

    private static final String TAG = makeLogTag(Header.class);

    // eager load
    private static final Header ourInstance = new Header();

    private static final String DEVICE = "android";
    // Defaut value : sweepin_android. If it doesn't work use proximities
    // private static final String APP_ID = "sweepin_android";
    private static final String API_VERSION = "3.0";

    private static final String APP_FIRST_LAUNCH = "1";

    public static Header getInstance() {
        return ourInstance;
    }

    private Header() { }

    /**
     * Construct a map to represent headers
     *
     * @param ctx       Context of the application
     * @param urlToSign String formatted like this ==> Method&url
     *                  such as "GET&http://google.com"
     * @return the minimal headers to send to the WS
     */
    private Map<String, String> constructHeaderBase(Context ctx, String urlToSign) {
        Map<String, String> params = new HashMap<>();
        params.put("device", DEVICE);
        params.put("language", Utils.getCurrentLanguage(ctx));
        try {
            params.put("signature", UrlSigner.sign(ctx, urlToSign));
        } catch (IOException | NoSuchAlgorithmException | InvalidKeyException | URISyntaxException e) {
            e.printStackTrace();
        }
        params.put("appid", Utils.getAppId(ctx));
        params.put("apiversion", API_VERSION);
        if(ProximitiesPrefs.readAppFirstLaunch(ctx).isEmpty()){
            ProximitiesPrefs.writeAppFirstLaunch(ctx, APP_FIRST_LAUNCH);
            params.put("appfirstlaunch", ProximitiesPrefs.readAppFirstLaunch(ctx));
        }
        if (!ProximitiesPrefs.readUserEmail(ctx).isEmpty()) {
            params.put("useridentifier", ProximitiesPrefs.readUserEmail(ctx));
        } else if (!ProximitiesPrefs.readAnonymousId(ctx).isEmpty()) {
            params.put("anonymousidentifier", ProximitiesPrefs.readAnonymousId(ctx));
        } else {
            ProximitiesPrefs.writeAnonymousId(ctx, UUID.randomUUID().toString());
            params.put("anonymousidentifier", ProximitiesPrefs.readAnonymousId(ctx));
        }
        return params;
    }

    /**
     * Construct a map to represent headers
     *
     * @param ctx       Context of the application
     * @param urlToSign String formatted like this ==> Method&url
     *                  such as "GET&http://google.com"
     * @return the minimal headers to send to the WS
     */
    private Map<String, String> constructHeaderBaseWithoutAnonymous(Context ctx, String urlToSign) {
        Map<String, String> params = new HashMap<>();
        params.put("device", DEVICE);
        params.put("language", Utils.getCurrentLanguage(ctx));
        try {
            params.put("signature", UrlSigner.sign(ctx, urlToSign));
        } catch (IOException | NoSuchAlgorithmException | InvalidKeyException | URISyntaxException e) {
            e.printStackTrace();
        }
        params.put("appid", Utils.getAppId(ctx));
        params.put("apiversion", API_VERSION);
        if(ProximitiesPrefs.readAppFirstLaunch(ctx).isEmpty()){
            ProximitiesPrefs.writeAppFirstLaunch(ctx, APP_FIRST_LAUNCH);
            params.put("appfirstlaunch", ProximitiesPrefs.readAppFirstLaunch(ctx));
        }
        if (!ProximitiesPrefs.readUserEmail(ctx).isEmpty()) {
            params.put("useridentifier", ProximitiesPrefs.readUserEmail(ctx));
        }  else {
            ProximitiesPrefs.writeAnonymousId(ctx, UUID.randomUUID().toString());
            params.put("anonymousidentifier", ProximitiesPrefs.readAnonymousId(ctx));
        }
        return params;
    }

    /**
     * Make necessary headers for the whole request of the app
     *
     * @param ctx       Context of the application
     * @param urlToSign String formatted like this ==> Method&url
     *                  such as "GET&http://google.com"
     * @return map which represents headers
     */
    public Map<String, String> makeHeader(Context ctx, String urlToSign) {
        String userModes = makeUserModes(ctx);
        String userCategories = makeUserCategories(ctx);
        Map<String, String> params;
        if (userModes == null || userModes.isEmpty()) {  // if there is no userModes
            params = constructHeaderBase(ctx, urlToSign);
        } else {
            params = makeHeader(ctx, urlToSign, userModes);
        }
        if (userCategories != null && !userCategories.isEmpty()){
            params.put("usercategories", userCategories);
        }
        LOGD(TAG, params.toString());
        return params;
    }

    /**
     * Make necessary headers for the whole request of the app
     *
     * @param ctx       Context of the application
     * @param urlToSign String formatted like this ==> Method&url
     *                  such as "GET&http://google.com"
     * @return map which represents headers
     */
    public Map<String, String> makeHeaderWithoutAnonymous(Context ctx, String urlToSign) {
        String userModes = makeUserModes(ctx);
        String userCategories = makeUserCategories(ctx);
        Map<String, String> params;
        if (userModes == null) {  // if there is no userModes
            params = constructHeaderBaseWithoutAnonymous(ctx, urlToSign);
        } else {
            params = makeHeader(ctx, urlToSign, userModes);
        }
        if (userCategories != null && !userCategories.isEmpty()){
            params.put("usercategories", userCategories);
        }
        LOGD(TAG, params.toString());
        return params;
    }

    /**
     * Build a string which represent user modes
     *
     * @param ctx context of the application
     * @return string formatted to be sent with headers
     */
    private String makeUserModes(Context ctx) {
        String userModes = "";
        if (ProximitiesPrefs.readHungry(ctx)) {
            userModes = ProximitiesConstants.HUNGRY + ';';
        }
        if (ProximitiesPrefs.readShopping(ctx)) {
            userModes += ProximitiesConstants.SHOPPING + ';';
        }
        if (ProximitiesPrefs.readTourist(ctx)) {
            userModes += ProximitiesConstants.TOURIST + ';';
        }
        if (ProximitiesPrefs.readThirsty(ctx)) {
            userModes += ProximitiesConstants.THIRSTY;
        }
        return userModes;
    }

    /**
     * Build a string which represent user categories
     *
     * @param ctx context of the application
     * @return string formatted to be sent with headers
     */
    private String makeUserCategories(Context ctx) {
        String userCategories = "";
        if (!ProximitiesPrefs.readCategories(ctx, true).isEmpty()) {
            userCategories = ProximitiesPrefs.readCategories(ctx, true);
        }
        return userCategories;
    }

    /**
     * Make necessary headers for the whole request of the app
     *
     * @param ctx       Context of the application
     * @param urlToSign String formatted like this ==> Method&url
     *                  such as "GET&http://google.com"
     * @param userModes List of string which mut be formatting as following : hungry;shopping
     * @return map which represents headers
     */
    public Map<String, String> makeHeader(Context ctx, String urlToSign, String userModes) {
        Map<String, String> params = constructHeaderBase(ctx, urlToSign);
        if (userModes != null && !userModes.isEmpty()) {
            params.put("usermodes", userModes);
        }
        LOGD(TAG, params.toString());
        return params;
    }
}
