/*
 * Copyright (C) 2016 Bartosz Schiller.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.github.barteksc.pdfviewpager.async;

import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.os.AsyncTask;
import android.widget.ImageView;

import com.shockwave.pdfium.PdfDocument;
import com.shockwave.pdfium.PdfiumCore;

import java.io.ByteArrayOutputStream;
import java.lang.ref.WeakReference;

public class PdfRenderPageAsyncTask extends AsyncTask<Void, Void, Void> {

    private final static int MAX_DIMENSION = 1800;

    private WeakReference<ImageView> imageViewRef;
    private Bitmap bitmap;
    private PdfiumCore pdfiumCore;
    private PdfDocument pdfDocument;
    private int position;
    private OnPdfPageRenderListener listener;

    public PdfRenderPageAsyncTask(ImageView imageView, int position, PdfiumCore pdfiumCore, PdfDocument pdfDocument) {
        imageViewRef = new WeakReference<>(imageView);
        this.position = position;
        this.pdfiumCore = pdfiumCore;
        this.pdfDocument = pdfDocument;
    }

    @Override
    protected Void doInBackground(Void... params) {
        pdfiumCore.openPage(pdfDocument, position);

        int width = pdfiumCore.getPageWidth(pdfDocument, position);
        int height = pdfiumCore.getPageHeight(pdfDocument, position);


        if (width > height && width > MAX_DIMENSION) {
            float aspectRatio = width /
                    (float) height;
            width = MAX_DIMENSION;
            height = Math.round(width / aspectRatio);
        } else if (height > MAX_DIMENSION) {
            float aspectRatio = height /
                    (float) width;
            height = MAX_DIMENSION;
            width = Math.round(height / aspectRatio);
        }

        Bitmap pageBitmap = Bitmap.createBitmap(width, height, Bitmap.Config.ARGB_8888);
        pdfiumCore.renderPageBitmap(pdfDocument, pageBitmap, position, 0, 0,
                width, height);

        ByteArrayOutputStream out = new ByteArrayOutputStream();
        pageBitmap.compress(Bitmap.CompressFormat.JPEG, 95, out);
        pageBitmap.recycle();

        BitmapFactory.Options ops = new BitmapFactory.Options();
        ops.inJustDecodeBounds = false;

        byte[] arr = out.toByteArray();

        bitmap = BitmapFactory.decodeByteArray(arr, 0, arr.length, ops);

        return null;
    }

    @Override
    protected void onPostExecute(Void o) {
        if (imageViewRef.get() != null) {
            imageViewRef.get().setImageBitmap(bitmap);
            if (listener != null) {
                listener.onPageRendered(position, imageViewRef.get(), bitmap);
            }
        }
    }

    @Override
    public void onCancelled(Void result) {
        if (bitmap != null) {
            bitmap.recycle();
            bitmap = null;
        }
    }

    public PdfRenderPageAsyncTask setOnPdfPageRenderListener(OnPdfPageRenderListener listener) {
        this.listener = listener;
        return this;
    }

    public interface OnPdfPageRenderListener {
        void onPageRendered(int position, ImageView imageView, Bitmap bitmap);
    }
}
