package com.emarsys.core.worker;

import android.os.Handler;
import android.os.Looper;

import com.emarsys.core.CoreCompletionHandler;
import com.emarsys.core.connection.ConnectionChangeListener;
import com.emarsys.core.connection.ConnectionState;
import com.emarsys.core.connection.ConnectionWatchDog;
import com.emarsys.core.queue.Queue;
import com.emarsys.core.request.RequestExpiredException;
import com.emarsys.core.request.RequestModel;
import com.emarsys.core.request.RestClient;
import com.emarsys.core.util.Assert;


public class DefaultWorker implements ConnectionChangeListener, Worker {

    Queue<RequestModel> queue;
    ConnectionWatchDog connectionWatchDog;
    boolean locked;
    CoreCompletionHandler coreCompletionHandler;
    RestClient restClient;
    Handler coreSdkHandler;
    Handler uiHandler;

    public DefaultWorker(Queue<RequestModel> queue, ConnectionWatchDog connectionWatchDog, Handler coreSdkHandler, CoreCompletionHandler coreCompletionHandler) {
        this(queue, connectionWatchDog, coreSdkHandler, coreCompletionHandler, new RestClient());
    }

    public DefaultWorker(Queue<RequestModel> queue, ConnectionWatchDog connectionWatchDog, Handler coreSdkHandler, CoreCompletionHandler coreCompletionHandler, RestClient restClient) {
        Assert.notNull(queue, "Queue must not be null!");
        Assert.notNull(connectionWatchDog, "ConnectionWatchDog must not be null!");
        Assert.notNull(coreSdkHandler, "Handler must not be null!");
        Assert.notNull(coreCompletionHandler, "CoreCompletionHandler must not be null!");
        Assert.notNull(restClient, "Restclient must not be null!");
        this.coreCompletionHandler = coreCompletionHandler;
        this.queue = queue;
        this.connectionWatchDog = connectionWatchDog;
        this.connectionWatchDog.registerReceiver(this);
        this.coreSdkHandler = coreSdkHandler;
        this.uiHandler = new Handler(Looper.getMainLooper());
        this.restClient = restClient;
    }

    @Override
    public void lock() {
        locked = true;
    }

    @Override
    public void unlock() {
        locked = false;
    }

    @Override
    public boolean isLocked() {
        return locked;
    }

    @Override
    public void run() {
        if (!isLocked() && connectionWatchDog.isConnected() && !queue.isEmpty()) {
            lock();
            RequestModel model = findFirstNonExpiredModel();
            if(model != null){
                restClient.execute(model, new CoreCompletionHandlerMiddleware(this, queue, coreSdkHandler, coreCompletionHandler));
            } else {
                unlock();
            }
        }
    }

    @Override
    public void onConnectionChanged(ConnectionState connectionState, boolean isConnected) {
        if (isConnected) {
            run();
        }
    }

    private RequestModel findFirstNonExpiredModel() {
        while (!queue.isEmpty()) {
            RequestModel model = queue.peek();
            if (isExpired(model)) {
                handleExpiration(model);
            } else {
                return model;
            }
        }
        return null;
    }

    private boolean isExpired(RequestModel model) {
        long now = System.currentTimeMillis();
        return now - model.getTimestamp() > model.getTtl();
    }

    private void handleExpiration(final RequestModel expiredModel) {
        queue.pop();
        uiHandler.post(new Runnable() {
            @Override
            public void run() {
                coreCompletionHandler.onError(expiredModel.getId(), new RequestExpiredException("Request expired", expiredModel.getUrl().getPath()));
            }
        });
    }
}