package com.emarsys.core.request;

import android.os.AsyncTask;

import com.emarsys.core.CoreCompletionHandler;
import com.emarsys.core.response.ResponseModel;

import org.json.JSONObject;

import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.List;
import java.util.Map;

import javax.net.ssl.HttpsURLConnection;


public class RequestTask extends AsyncTask<Void, Long, Void> {
    private static final int TIMEOUT = 30000;

    private final RequestModel requestModel;
    private final CoreCompletionHandler handler;

    private ResponseModel responseModel;
    private Exception exception;

    public RequestTask(RequestModel requestModel, CoreCompletionHandler handler) {
        this.requestModel = requestModel;
        this.handler = handler;
    }

    @Override
    protected Void doInBackground(Void... params) {
        HttpsURLConnection connection = null;
        try {
            connection = (HttpsURLConnection) requestModel.getUrl().openConnection();
            initializeConnection(connection, requestModel);
            connection.connect();
            sendBody(connection, requestModel);
            responseModel = readResponse(connection);
        } catch (IOException ioe) {
            exception = ioe;
        } finally {
            if (connection != null) {
                connection.disconnect();
            }
        }
        return null;
    }

    @Override
    protected void onPostExecute(Void result) {
        if (exception != null) {
            handler.onError(requestModel.getId(), exception);

        } else if (responseModel != null) {
            if (isStatusCodeOK(responseModel.getStatusCode())) {
                handler.onSuccess(requestModel.getId(), responseModel);
            } else {
                handler.onError(requestModel.getId(), responseModel);
            }
        }
    }

    public RequestModel getRequestModel() {
        return requestModel;
    }

    public CoreCompletionHandler getHandler() {
        return handler;
    }

    private void initializeConnection(HttpsURLConnection connection, RequestModel model) throws IOException {
        connection.setRequestMethod(model.getMethod().name());
        setHeaders(connection, model.getHeaders());
        connection.setConnectTimeout(TIMEOUT);
        if (model.getMethod() != RequestMethod.GET && model.getPayload() != null) {
            connection.setDoOutput(true);
        }
    }

    private void setHeaders(HttpsURLConnection connection, Map<String, String> headers) {
        for (Map.Entry<String, String> entry : headers.entrySet()) {
            String key = entry.getKey();
            String value = entry.getValue();
            connection.setRequestProperty(key, value);
        }
    }

    private void sendBody(HttpsURLConnection connection, RequestModel model) throws IOException {
        if (model.getPayload() != null) {
            byte[] payload = new JSONObject(model.getPayload()).toString().getBytes("UTF-8");
            BufferedOutputStream writer = new BufferedOutputStream(connection.getOutputStream());
            writer.write(payload);
            writer.close();
        }
    }

    private ResponseModel readResponse(HttpsURLConnection connection) throws IOException {
        int statusCode = connection.getResponseCode();
        String message = connection.getResponseMessage();
        Map<String, List<String>> headers = connection.getHeaderFields();
        String body = readBody(connection);
        return new ResponseModel.Builder().statusCode(statusCode).message(message).headers(headers).body(body).build();
    }

    private String readBody(HttpsURLConnection connection) throws IOException {
        int responseCode = connection.getResponseCode();
        InputStream inputStream;
        if (isStatusCodeOK(responseCode)) {
            inputStream = connection.getInputStream();
        } else {
            inputStream = connection.getErrorStream();
        }
        BufferedReader reader = new BufferedReader(new InputStreamReader(inputStream));
        String inputLine;
        StringBuilder sb = new StringBuilder();

        while ((inputLine = reader.readLine()) != null) {
            sb.append(inputLine);
        }

        reader.close();
        return sb.toString();
    }

    private boolean isStatusCodeOK(int responseCode) {
        return 200 <= responseCode && responseCode < 300;
    }
}
