package com.emarsys.core.worker;

import android.os.AsyncTask;
import android.os.Handler;
import android.os.Looper;

import com.emarsys.core.CoreCompletionHandler;
import com.emarsys.core.connection.ConnectionChangeListener;
import com.emarsys.core.connection.ConnectionState;
import com.emarsys.core.connection.ConnectionWatchDog;
import com.emarsys.core.queue.Queue;
import com.emarsys.core.request.RequestModel;
import com.emarsys.core.request.factory.DefaultRequestTaskFactory;
import com.emarsys.core.request.factory.RequestTaskFactory;
import com.emarsys.core.util.Assert;


public class DefaultWorker implements ConnectionChangeListener, Worker {

    Queue<RequestModel> queue;
    ConnectionWatchDog connectionWatchDog;
    boolean locked;
    CoreCompletionHandler coreCompletionHandler;
    RequestTaskFactory taskFactory;

    private final Handler handler = new Handler(Looper.getMainLooper());

    public DefaultWorker(Queue<RequestModel> queue, ConnectionWatchDog connectionWatchDog, CoreCompletionHandler coreCompletionHandler) {
        this(queue, connectionWatchDog, coreCompletionHandler, new DefaultRequestTaskFactory());
    }

    public DefaultWorker(Queue<RequestModel> queue, ConnectionWatchDog connectionWatchDog, CoreCompletionHandler coreCompletionHandler, RequestTaskFactory taskFactory) {
        Assert.notNull(queue, "Queue must not be null!");
        Assert.notNull(connectionWatchDog, "ConnectionWatchDog must not be null!");
        Assert.notNull(coreCompletionHandler, "CoreCompletionHandler must not be null!");
        this.coreCompletionHandler = coreCompletionHandler;
        this.queue = queue;
        this.connectionWatchDog = connectionWatchDog;
        this.connectionWatchDog.registerReceiver(this);
        this.taskFactory = taskFactory;
    }

    @Override
    public void lock() {
        locked = true;
    }

    @Override
    public void unlock() {
        locked = false;
    }

    @Override
    public boolean isLocked() {
        return locked;
    }

    @Override
    public void run() {
        if (!isLocked()) {
            lock();
            execute();
        }
    }

    void execute() {
        if (connectionWatchDog.isConnected() && !queue.isEmpty()) {
            RequestModel model = queue.peek();
            sendRequest(model, new CoreCompletionHandlerMiddleware(this, queue, coreCompletionHandler));
        } else {
            unlock();
        }
    }

    void sendRequest(final RequestModel model, final CoreCompletionHandler completionHandler) {
        if (Looper.myLooper() == Looper.getMainLooper()) {
            taskFactory.create(model, completionHandler).executeOnExecutor(AsyncTask.SERIAL_EXECUTOR);
        } else {
            handler.post(new Runnable() {
                @Override
                public void run() {
                    taskFactory.create(model, completionHandler).executeOnExecutor(AsyncTask.SERIAL_EXECUTOR);
                }
            });
        }
    }

    @Override
    public void onConnectionChanged(ConnectionState connectionState, boolean isConnected) {
        if (isConnected) {
            run();
        }
    }
}