package com.emarsys.core.queue.sqlite;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;

import com.emarsys.core.queue.Queue;
import com.emarsys.core.request.RequestMethod;
import com.emarsys.core.request.RequestModel;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.Date;
import java.util.Map;

import static com.emarsys.core.queue.sqlite.RequestContract.COLUMN_NAME_HEADERS;
import static com.emarsys.core.queue.sqlite.RequestContract.COLUMN_NAME_METHOD;
import static com.emarsys.core.queue.sqlite.RequestContract.COLUMN_NAME_PAYLOAD;
import static com.emarsys.core.queue.sqlite.RequestContract.COLUMN_NAME_REQUEST_ID;
import static com.emarsys.core.queue.sqlite.RequestContract.COLUMN_NAME_TIMESTAMP;
import static com.emarsys.core.queue.sqlite.RequestContract.COLUMN_NAME_URL;
import static com.emarsys.core.queue.sqlite.RequestContract.SQL_COUNT;
import static com.emarsys.core.queue.sqlite.RequestContract.SQL_DELETE_ITEM;
import static com.emarsys.core.queue.sqlite.RequestContract.SQL_SELECTFIRST;
import static com.emarsys.core.queue.sqlite.RequestContract.TABLE_NAME;

public class SqliteQueue implements Queue<RequestModel> {

    SQLiteOpenHelper helper;

    public SqliteQueue(Context context) {
        helper = new DbHelper(context);
    }

    @Override
    public void push(RequestModel item) {
        ContentValues contentValues = new ContentValues();
        contentValues.put(COLUMN_NAME_REQUEST_ID, item.getId());
        contentValues.put(COLUMN_NAME_METHOD, item.getMethod().name());
        contentValues.put(COLUMN_NAME_URL, item.getUrl().toString());
        contentValues.put(COLUMN_NAME_HEADERS, serializableToBlob(item.getHeaders()));
        contentValues.put(COLUMN_NAME_PAYLOAD, serializableToBlob(item.getPayload()));
        contentValues.put(COLUMN_NAME_TIMESTAMP, item.getTimestamp().getTime());

        SQLiteDatabase database = helper.getWritableDatabase();
        database.beginTransaction();
        database.insert(TABLE_NAME, null, contentValues);
        database.setTransactionSuccessful();
        database.endTransaction();
    }

    @Override
    public RequestModel pop() {
        SQLiteDatabase db = helper.getWritableDatabase();
        db.beginTransaction();

        RequestModel model = getFirst(db);

        if (model != null) {
            db.execSQL(String.format(SQL_DELETE_ITEM, model.getId()));
        }

        db.setTransactionSuccessful();
        db.endTransaction();
        return model;

    }

    @Override
    public RequestModel peek() {
        return getFirst(helper.getReadableDatabase());

    }

    @Override
    public boolean isEmpty() {
        Cursor cursor = helper.getReadableDatabase().rawQuery(SQL_COUNT, null);
        cursor.moveToFirst();
        int count = cursor.getInt(0);
        cursor.close();
        return count == 0;
    }

    public void setHelper(SQLiteOpenHelper helper) {
        this.helper = helper;
    }

    private RequestModel getFirst(SQLiteDatabase db) {
        RequestModel requestModel = null;

        Cursor cursor = db.rawQuery(SQL_SELECTFIRST, null);
        if (cursor.getCount() != 0) {
            cursor.moveToFirst();

            String requestId = cursor.getString(cursor.getColumnIndex(COLUMN_NAME_REQUEST_ID));

            RequestMethod method = RequestMethod.valueOf(cursor.getString(cursor.getColumnIndex(COLUMN_NAME_METHOD)));

            String url = cursor.getString(cursor.getColumnIndex(COLUMN_NAME_URL));

            Map<String, String> headers = blobToSerializable(cursor.getBlob(cursor.getColumnIndex(COLUMN_NAME_HEADERS)));

            Map<String, Object> payload = blobToSerializable(cursor.getBlob(cursor.getColumnIndex(COLUMN_NAME_PAYLOAD)));

            Date timeStamp = new Date(cursor.getLong(cursor.getColumnIndex(COLUMN_NAME_TIMESTAMP)));

            requestModel = new RequestModel(url, method, payload, headers, timeStamp, requestId);
        }
        cursor.close();

        return requestModel;
    }

    private byte[] serializableToBlob(Object object) {
        try {
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            ObjectOutputStream oos = new ObjectOutputStream(baos);
            oos.writeObject(object);
            oos.close();
            return baos.toByteArray();
        } catch (IOException e) {
            throw new RuntimeException("Exception while converting object to blob", e);
        }
    }

    private <T> T blobToSerializable(byte[] blob) {
        try {
            ByteArrayInputStream bais = new ByteArrayInputStream(blob);
            ObjectInputStream ois = new ObjectInputStream(bais);
            bais.close();
            return (T) ois.readObject();
        } catch (IOException | ClassNotFoundException e) {
            throw new RuntimeException("Exception while converting blob to object.", e);
        }
    }
}
