package com.emarsys.core.response;

import com.emarsys.core.util.Assert;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class ResponseModel {

    private final int statusCode;
    private final String message;
    private final Map<String, String> headers;
    private final String body;

    ResponseModel(int statusCode, String message, Map<String, String> headers, String body) {
        checkStatusCode(statusCode);
        Assert.notNull(message, "Message must not be null!");
        Assert.notNull(headers, "Headers must not be null!");
        this.statusCode = statusCode;
        this.message = message;
        this.headers = headers;
        this.body = body;
    }

    private void checkStatusCode(int statusCode) {
        if(statusCode < 200 || statusCode >= 600) {
            throw new IllegalArgumentException("Status code must not be null!");
        }
    }

    public int getStatusCode() {
        return statusCode;
    }

    public String getMessage() {
        return message;
    }

    public Map<String, String> getHeaders() {
        return headers;
    }

    public String getBody() {
        return body;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        ResponseModel that = (ResponseModel) o;

        if (getStatusCode() != that.getStatusCode()) return false;
        if (!getMessage().equals(that.getMessage())) return false;
        if (!getHeaders().equals(that.getHeaders())) return false;
        return getBody() != null ? getBody().equals(that.getBody()) : that.getBody() == null;

    }

    @Override
    public int hashCode() {
        int result = getStatusCode();
        result = 31 * result + getMessage().hashCode();
        result = 31 * result + getHeaders().hashCode();
        result = 31 * result + (getBody() != null ? getBody().hashCode() : 0);
        return result;
    }

    public static class Builder {
        private int statusCode;
        private String message;
        private Map<String, String> headers;
        private String body;

        public Builder() {
            this.headers = new HashMap<>();
        }

        public Builder statusCode(int statusCode) {
            this.statusCode = statusCode;
            return this;
        }

        public Builder message(String message) {
            this.message = message;
            return this;
        }

        public Builder headers(Map<String, List<String>> headers) {
            this.headers = convertHeaders(headers);
            return this;
        }

        public Builder body(String body) {
            this.body = body;
            return this;
        }

        public ResponseModel build() {
            return new ResponseModel(statusCode, message, headers, body);
        }

        Map<String, String> convertHeaders(Map<String, List<String>> headers) {
            Map<String, String> result = new HashMap<>();
            Set<Map.Entry<String, List<String>>> entries = headers.entrySet();

            for (Map.Entry<String, List<String>> entry : entries) {
                result.put(entry.getKey(), join("; ", entry.getValue()));
            }
            return result;
        }

        String join(String delimiter, List<String> strings) {
            StringBuilder stringBuilder = new StringBuilder();
            Iterator<String> iterator = strings.iterator();

            if (iterator.hasNext()) {
                stringBuilder.append(iterator.next());
                while (iterator.hasNext()) {
                    stringBuilder.append(delimiter);
                    stringBuilder.append(iterator.next());
                }
            }
            return stringBuilder.toString();
        }
    }

}
