package com.emarsys.core.request;

import com.emarsys.core.util.Assert;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

public class RequestModel {
    private final URL url;
    private final RequestMethod method;
    private final Map<String, Object> payload;
    private final Map<String, String> headers;
    private final Date timestamp;
    private final String id;

    RequestModel(String url, RequestMethod method, Map<String, Object> payload, Map<String, String> headers, Date timestamp, String id) {
        Assert.notNull(url, "Url must not be null!");
        Assert.notNull(method, "Method must not be null!");
        Assert.notNull(headers, "Headers must not be null!");
        Assert.notNull(timestamp, "Timestamp must not be null!");
        Assert.notNull(id, "Id must not be null!");
        try{
            this.url = new URL(url);
        }catch (MalformedURLException mue){
            throw new IllegalArgumentException(mue);
        }
        this.method = method;
        this.payload = payload;
        this.headers = headers;
        this.timestamp = timestamp;
        this.id = id;
    }

    public static String nextId() {
        return UUID.randomUUID().toString();
    }

    public URL getUrl() {
        return url;
    }

    public RequestMethod getMethod() {
        return method;
    }

    public Map<String, Object> getPayload() {
        return payload;
    }

    public Map<String, String> getHeaders() {
        return headers;
    }

    public Date getTimestamp() {
        return timestamp;
    }

    public String getId() {
        return id;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        RequestModel that = (RequestModel) o;

        if (!getUrl().equals(that.getUrl())) return false;
        if (getMethod() != that.getMethod()) return false;
        if (getPayload() != null ? !getPayload().equals(that.getPayload()) : that.getPayload() != null)
            return false;
        if (getHeaders() != null ? !getHeaders().equals(that.getHeaders()) : that.getHeaders() != null)
            return false;
        if (getTimestamp() != null ? !getTimestamp().equals(that.getTimestamp()) : that.getTimestamp() != null)
            return false;
        return getId() != null ? getId().equals(that.getId()) : that.getId() == null;
    }

    @Override
    public int hashCode() {
        int result = getUrl().hashCode();
        result = 31 * result + (getMethod() != null ? getMethod().hashCode() : 0);
        result = 31 * result + (getPayload() != null ? getPayload().hashCode() : 0);
        result = 31 * result + (getHeaders() != null ? getHeaders().hashCode() : 0);
        result = 31 * result + (getTimestamp() != null ? getTimestamp().hashCode() : 0);
        result = 31 * result + (getId() != null ? getId().hashCode() : 0);
        return result;
    }

    public static class Builder {
        private String url;
        private RequestMethod method;
        private Map<String, Object> payload;
        private Map<String, String> headers;
        private Date timestamp;
        private String id;

        public Builder(){
            method = RequestMethod.POST;
            headers = new HashMap<>();
            timestamp = new Date();
            id = RequestModel.nextId();
        }

        public Builder url(String url) {
            this.url = url;
            return this;
        }

        public Builder method(RequestMethod method) {
            this.method = method;
            return this;
        }

        public Builder payload(Map<String, Object> payload) {
            this.payload = payload;
            return this;
        }

        public Builder headers(Map<String, String> headers) {
            this.headers = headers;
            return this;
        }

        public RequestModel build() {
            return new RequestModel(url, method, payload, headers, timestamp, id);
        }
    }
}