package com.emarsys.core.worker;

import android.os.Handler;
import android.os.Looper;

import com.emarsys.core.CoreCompletionHandler;
import com.emarsys.core.queue.Queue;
import com.emarsys.core.request.RequestModel;
import com.emarsys.core.request.factory.DefaultRunnableFactory;
import com.emarsys.core.request.factory.RunnableFactory;
import com.emarsys.core.response.ResponseModel;
import com.emarsys.core.util.Assert;
import com.emarsys.core.util.log.CoreTopic;
import com.emarsys.core.util.log.EMSLogger;

public class CoreCompletionHandlerMiddleware implements CoreCompletionHandler {
    CoreCompletionHandler coreCompletionHandler;
    Queue<RequestModel> queue;
    Worker worker;
    Handler coreSDKHandler;
    Handler uiHandler;
    RunnableFactory runnableFactory;

    public CoreCompletionHandlerMiddleware(final Worker worker, Queue<RequestModel> queue, Handler coreSDKHandler, CoreCompletionHandler coreCompletionHandler) {
        Assert.notNull(queue, "Queue must not be null!");
        Assert.notNull(worker, "Worker must not be null!");
        Assert.notNull(coreCompletionHandler, "CoreCompletionHandler must not be null!");
        Assert.notNull(coreSDKHandler, "coreSDKHandler must not be null!");
        this.coreCompletionHandler = coreCompletionHandler;
        this.queue = queue;
        this.worker = worker;
        this.coreSDKHandler = coreSDKHandler;
        this.runnableFactory = new DefaultRunnableFactory();
        this.uiHandler = new Handler(Looper.getMainLooper());
    }

    @Override
    public void onSuccess(final String id, final ResponseModel responseModel) {
        EMSLogger.log(CoreTopic.OFFLINE, "Id: %s, response model: %s", id, responseModel);

        coreSDKHandler.post(runnableFactory.runnableFrom(new Runnable() {
            @Override
            public void run() {
                queue.pop();
                worker.unlock();
                worker.run();

                uiHandler.post(runnableFactory.runnableFrom(new Runnable() {
                    @Override
                    public void run() {
                        coreCompletionHandler.onSuccess(id, responseModel);
                    }
                }));
            }
        }));
    }

    @Override
    public void onError(final String id, final ResponseModel responseModel) {
        EMSLogger.log(CoreTopic.OFFLINE, "Id: %s, response model: %s", id, responseModel);

        coreSDKHandler.post(runnableFactory.runnableFrom(new Runnable() {
            @Override
            public void run() {
                if (isNonRetriableError(responseModel.getStatusCode())) {
                    queue.pop();
                    worker.unlock();
                    worker.run();
                } else {
                    worker.unlock();
                }

                uiHandler.post(runnableFactory.runnableFrom(new Runnable() {
                    @Override
                    public void run() {
                        if (isNonRetriableError(responseModel.getStatusCode())) {
                            coreCompletionHandler.onError(id, responseModel);
                        }
                    }
                }));
            }
        }));
    }

    @Override
    public void onError(final String id, final Exception cause) {
        EMSLogger.log(CoreTopic.OFFLINE, "Id: %s, exception: %s", id, cause);

        coreSDKHandler.post(runnableFactory.runnableFrom(new Runnable() {
            @Override
            public void run() {
                worker.unlock();
                uiHandler.post(runnableFactory.runnableFrom(new Runnable() {
                    @Override
                    public void run() {
                        coreCompletionHandler.onError(id, cause);
                    }
                }));
            }
        }));
    }

    private boolean isNonRetriableError(int statusCode) {
        if (statusCode == 408) {
            return false;
        } else {
            return 400 <= statusCode && statusCode < 500;
        }
    }
}
