package com.citruspay.citrusbrowser.views;

import android.app.Activity;
import android.content.Context;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.os.CountDownTimer;
import android.os.Handler;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.util.Log;
import android.view.View;
import android.view.animation.Animation;
import android.view.animation.AnimationUtils;
import android.widget.Button;
import android.widget.CheckBox;
import android.widget.CompoundButton;
import android.widget.EditText;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;

import com.citruspay.citrusbrowser.R;
import com.citruspay.citrusbrowser.core.BankJSConfig;
import com.citruspay.citrusbrowser.core.CitrusBrowserConfig;
import com.citruspay.citrusbrowser.core.FlashConfigParser;
import com.citruspay.citrusbrowser.core.PaymentStatus;
import com.citruspay.citrusbrowser.listener.OTPViewListener;


/**
 * This class handles the OTPPopup View and all its event listeners.
 * Created by Gautam on 26/3/16.
 */
public class OTPPopupView extends LinearLayout implements View.OnClickListener, ProgressGenerator.OnCompleteListener {

    private static final long RESEND_BTN_VISIBLE_TIMER = 30000;
    private static final long RESEND_BTN_TICK_TIMER = 1000;
    private boolean enterOtpYourself = false;
    private boolean autoSubmitAllowed = true;
    private boolean smsPermissionGranted = false;
    private CountDownTimer mStartResendButtonVisibilityTimer;
    private boolean mResendButtonVisibilityTimerRunning = false;

    public boolean isResendClicked() {
        return resendClicked;
    }

    private boolean resendClicked = false;
    private String userEnteredOtp = null;
    private String userEnteredPass = null;
    private String autoReadOTP = "";
    private Context context;
    private OTPViewListener listener;
    private TextView continueWithPassText = null;
    private EditText enterOtpEditTxt = null;
    private Button otpResendBtn = null;
    private EditText enterPassEdTxt = null;
    private Animation fadeIn = null;
    private ProgressGenerator progressGenerator = null;
    private ActionProcessButton btnRequestOTP = null;
    private ActionProcessButton otpConfirmBtn = null;
    private ActionProcessButton btnSubmitPass = null;
    private OTPAutoReadState otpAutoReadState = OTPAutoReadState.None;
    private FlashConfigParser flashConfig;
    private BankJSConfig bankJsConfig;

    public OTPPopupView(Context context) {
        super(context);
        this.context = context;
        init(null, 0);
    }

    public OTPPopupView(Context context, AttributeSet attrs) {
        super(context, attrs);
        this.context = context;
        init(attrs, 0);
    }

    public void smsPermissionStatus(boolean smsPermissionStatus) {
        smsPermissionGranted = smsPermissionStatus;

        if (smsPermissionGranted) {
            setOtpAutoReadState(OTPAutoReadState.AutoReadEnabled);
        } else {
            setOtpAutoReadState(OTPAutoReadState.AutoReadDisabled);
        }
    }

    public void setFlashConfig(FlashConfigParser flashConfig) {
        this.flashConfig = flashConfig;
    }

    public void setBankJsConfig(BankJSConfig bankJsConfig) {
        this.bankJsConfig = bankJsConfig;
    }

    public enum OTPAutoReadState {
        AutoReadEnabled,
        AutoReadDisabled,
        None
    }

    private void init(AttributeSet attrs, int defStyle) {

        fadeIn = AnimationUtils.loadAnimation(context,
                R.anim.fade_in);

        inflate(context, R.layout.otppopup_txn_options, this);


        progressGenerator = new ProgressGenerator(this);

        btnRequestOTP = (ActionProcessButton) findViewById(R.id.requestOTPBtnId);
        btnRequestOTP.setMode(ActionProcessButton.Mode.PROGRESS);
        btnRequestOTP.setOnClickNormalState(new OnClickListener() {
            @Override
            public void onClick(View v) {

                KeyBoardUtil.hideKeyboard((Activity) context);

                progressGenerator.setRandomDelayRange(500);
                progressGenerator.start(btnRequestOTP);
                listener.onSendOtpClicked();
                btnRequestOTP.setClickable(false);
            }
        }).build();


        continueWithPassText = (TextView) findViewById(R.id.continuePasswordTextViewId);
        continueWithPassText.setOnClickListener(this);
        CheckBox rememberMeCheckBox = (CheckBox) findViewById(R.id.rememberMeCheckBoxId);

        rememberMeCheckBox.setOnCheckedChangeListener(new CompoundButton.OnCheckedChangeListener() {
            @Override
            public void onCheckedChanged(CompoundButton buttonView, boolean isChecked) {

            }
        });

        mStartResendButtonVisibilityTimer = new CountDownTimer(RESEND_BTN_VISIBLE_TIMER, RESEND_BTN_TICK_TIMER) {

            public void onTick(long millisUntilFinished) {
                mResendButtonVisibilityTimerRunning = true;

                otpResendBtn.setText("" + millisUntilFinished / 1000);
            }

            public void onFinish() {
                mResendButtonVisibilityTimerRunning = false;

                otpResendBtn.setText("Resend");
                otpResendBtn.setEnabled(true);
                otpResendBtn.setClickable(true);
            }
        };
    }

    /**
     * Set Relevant Card Password text
     *
     * @param password
     */
    public void setPasswordText(String password) {
        if (continueWithPassText != null) {
            continueWithPassText.setText(password);
        }
    }

    /**
     * Set Relevant Card Password hint
     *
     * @param passwordHint
     */
    public void setPasswordHintext(String passwordHint) {
        if (enterPassEdTxt != null) {
            enterPassEdTxt.setHint(passwordHint);
        }
    }

    /**
     * OTP Read timeout scenario.
     */
    public void otpReadTimeout() {

        Log.d("OTP READ", "Timeout");

        removeAllViewsInLayout();
        inflate(context, R.layout.otp_autoread_timeout, this);

        if (findViewById(R.id.otpWaitingLayoutId) != null)
            findViewById(R.id.otpWaitingLayoutId).setVisibility(View.INVISIBLE);

        if (findViewById(R.id.otpTimeoutLayoutId) != null) {
            this.otpResendBtn = (Button) findViewById(R.id.otpResendBtnId);
            this.otpResendBtn.setOnClickListener(this);
            findViewById(R.id.otpTimeoutLayoutId).setVisibility(View.VISIBLE);
            this.otpResendBtn.setEnabled(true);
            this.otpResendBtn.setClickable(true);


            this.enterOtpEditTxt = (EditText) findViewById(R.id.enterOtpEditTextId);
            this.enterOtpEditTxt.requestFocus();

            this.otpConfirmBtn = (ActionProcessButton) findViewById(R.id.otpConfirmBtnId);
            this.otpConfirmBtn.setMode(ActionProcessButton.Mode.PROGRESS);
            this.otpConfirmBtn.setOnClickNormalState(new OnClickListener() {
                @Override
                public void onClick(View v) {
                    KeyBoardUtil.hideKeyboard((Activity) context);
                    userEnteredOtp = enterOtpEditTxt.getText().toString();
                    if (TextUtils.isEmpty(userEnteredOtp)) {
                        // Otp detection failed or timeout and no otp entered
                        enterOtpEditTxt.requestFocus();
                        enterOtpEditTxt.setError("Please enter OTP or click Resend");
                    } else if (userEnteredOtp.length() < 6) {
                        // Otp entered is less than 6 digits.
                        enterOtpEditTxt.requestFocus();
                        enterOtpEditTxt.setError("Please enter a 6 digit OTP");
                    } else {
                        // Otp detection failed or timeout and user entered it manually
                        progressGenerator.setRandomDelayRange(900);
                        progressGenerator.start(otpConfirmBtn);
                        otpConfirmBtn.setClickable(false);
                        listener.onProcessTransactionClicked(userEnteredOtp);
                    }

                }
            }).build();


            // This happens if user chooses to enter otp manually,
            // or if SMS permission is denied.
            if (enterOtpYourself) {
                otpResendBtn.setEnabled(false);
                otpResendBtn.setClickable(false);

                startResendButtonVisibilityTimer();
            }

            // If auto Submit OTP is not allowed.
            if (!autoSubmitAllowed) {
                this.otpResendBtn.setVisibility(View.GONE);
                this.enterOtpEditTxt.setText(autoReadOTP);
            }

            if (bankJsConfig.getHideResendBtn()) {
                this.otpResendBtn.setVisibility(View.GONE);
            }

        }
    }

    /**
     * Toggle enter password option for few banks.
     *
     * @param enabled
     */
    public void toggleEnterPasswordOption(boolean enabled) {
        if (findViewById(R.id.continuePasswordTextViewId) != null && !enabled) {
            findViewById(R.id.continuePasswordTextViewId).setVisibility(GONE);
        }
        if (findViewById(R.id.orTextId) != null && !enabled) {
            findViewById(R.id.orTextId).setVisibility(GONE);
        }
    }

    /**
     * Display Enter Password Popup
     */
    public void displayEnterPasswordPopup() {
        removeAllViewsInLayout();
        inflate(context, R.layout.otppopup_enter_password, this);

        btnSubmitPass = (ActionProcessButton) findViewById(R.id.submitPasswordBtnId);

        btnSubmitPass.setMode(ActionProcessButton.Mode.PROGRESS);
        btnSubmitPass.setOnClickNormalState(new OnClickListener() {
            @Override
            public void onClick(View v) {
                KeyBoardUtil.hideKeyboard((Activity) context);
                userEnteredPass = enterPassEdTxt.getText().toString();
                if (TextUtils.isEmpty(userEnteredPass)) {
                    // no password entered
                    enterPassEdTxt.requestFocus();
                    enterPassEdTxt.setError(enterPassEdTxt.getHint());
                } else {
                    // Handle user entered password
                    listener.onEnteredPasswordConfirmed(userEnteredPass);
                    progressGenerator.setRandomDelayRange(850);
                    progressGenerator.start(btnSubmitPass);
                    btnSubmitPass.setClickable(false);
                }
            }
        }).build();


        enterPassEdTxt = (EditText) findViewById(R.id.enterPasswordEditTextId);
        enterPassEdTxt.requestFocus();
        setPasswordHintext("Enter your " + flashConfig.getCardPwds().getCurrentCardPwd());
    }

    /**
     * Display AutoRead OTP flow.
     */
    public void displayAutoReadOTP() {

        if (findViewById(R.id.otpWaitingLayoutId) != null)
            findViewById(R.id.otpWaitingLayoutId).setVisibility(View.GONE);
        removeAllViewsInLayout();
        inflate(context, R.layout.otp_autoread, this);

        if (findViewById(R.id.otpWaitingLayoutId) != null)
            findViewById(R.id.otpWaitingLayoutId).setVisibility(View.GONE);
        if (findViewById(R.id.otpAutoReadLayoutId) != null) {
            findViewById(R.id.otpAutoReadLayoutId).setVisibility(View.VISIBLE);
            ((TypeWriterTextView) findViewById(R.id.otpAutoReadHeaderTxtId)).animateText("Auto Reading OTP", 0);
        }
    }

    /**
     * Display Auto Read Set OTP flow.
     *
     * @param otp
     */
    public void displayAutoReadSetOTP(final String otp) {

        if (findViewById(R.id.otpWaitingLayoutId) != null) {
            findViewById(R.id.otpWaitingLayoutId).setVisibility(View.GONE);
        }
        TypeWriterTextView otpAutoRead = (TypeWriterTextView) findViewById(R.id.otpAutoReadTxtId);
        if (otpAutoRead != null) {
            otpAutoRead.setCharacterDelay(500);
            otpAutoRead.animateText(otp, 200);
        }

        autoSubmitAllowed = CitrusBrowserConfig.getInstance().isAutoSubmitOTP();

        if (autoSubmitAllowed) {
            // Auto Submit OTP is allowed.
            final Handler handler = new Handler();
            handler.postDelayed(new Runnable() {
                @Override
                public void run() {
                    displaySubmitOTP(otp);
                }
            }, 2500);
        } else {
            // Auto Submit OTP is not allowed.
            autoReadOTP = otp;
            otpReadTimeout();

        }
    }

    /**
     * Display Submit OTP flow.
     *
     * @param otp
     */
    public void displaySubmitOTP(String otp) {

        if (findViewById(R.id.otpWaitingLayoutId) != null) {
            findViewById(R.id.otpWaitingLayoutId).setVisibility(View.GONE);
        }

        removeAllViewsInLayout();
        inflate(context, R.layout.otp_submit, this);

        if (findViewById(R.id.otpAutoReadLayoutId) != null)
            findViewById(R.id.otpAutoReadLayoutId).setVisibility(View.GONE);
        if (findViewById(R.id.otpSubmitLayoutId) != null) {
            findViewById(R.id.otpSubmitLayoutId).setVisibility(View.VISIBLE);
            ((TypeWriterTextView) findViewById(R.id.otpSubmittingHeaderTxtId)).animateText("Submitting OTP", 0);
            TypeWriterTextView otpAutoSubmit = (TypeWriterTextView) findViewById(R.id.otpSubmitTxtId);
            otpAutoSubmit.setCharacterDelay(500);
            otpAutoSubmit.animateText(otp, 200);

            DilatingDotsProgressBar mDilatingDotsProgressBar = (DilatingDotsProgressBar) findViewById(R.id.submitProgress);

            // show progress bar and start animating
            mDilatingDotsProgressBar.show(1000);
        }
        if (findViewById(R.id.otpWaitingLayoutId) != null) {
            findViewById(R.id.otpWaitingLayoutId).setVisibility(View.GONE);
        }
    }

    /**
     * Manage view visibility on txn completion.
     */
    private void manageViewVisibility() {
        if (findViewById(R.id.otpSubmitLayoutId) != null)
            findViewById(R.id.otpSubmitLayoutId).setVisibility(View.GONE);
        if (findViewById(R.id.enterPasswordLayoutId) != null)
            findViewById(R.id.enterPasswordLayoutId).setVisibility(View.GONE);
        if (findViewById(R.id.otpTxnOptionsLayoutId) != null)
            findViewById(R.id.otpTxnOptionsLayoutId).setVisibility(View.GONE);
        if (findViewById(R.id.otpAutoReadLayoutId) != null)
            findViewById(R.id.otpAutoReadLayoutId).setVisibility(View.GONE);
        if (findViewById(R.id.otpWaitingLayoutId) != null)
            findViewById(R.id.otpWaitingLayoutId).setVisibility(View.GONE);
        if (findViewById(R.id.otpTimeoutLayoutId) != null)
            findViewById(R.id.otpTimeoutLayoutId).setVisibility(View.GONE);
        if (findViewById(R.id.otpRegisterCardTxtId) != null)
            findViewById(R.id.otpRegisterCardTxtId).setVisibility(View.GONE);
    }

    /**
     * Handle Cancel Txn UI.
     */
    public void handleCancelTransaction() {

        removeAllViewsInLayout();
        inflate(context, R.layout.otppopup_cancel_transaction, this);

        ((TypeWriterTextView) findViewById(R.id.otpCancelTransactionTxtId)).animateText("Cancelling Transaction...", 5);

        DilatingDotsProgressBar mDilatingDotsProgressBar = (DilatingDotsProgressBar) findViewById(R.id.progress);

        // show progress bar and start animating
        mDilatingDotsProgressBar.show(700);

        manageViewVisibility();
    }

    /**
     * Display Cancel Txn UI.
     *
     * @param paymentStatus
     */
    //    public void displayTransactionStatus(/*TransactionResponse transactionResponse*/) {
    public void displayTransactionStatus(PaymentStatus paymentStatus) {
        removeAllViewsInLayout();
        inflate(context, R.layout.otppopup_transaction_successfull, this);

        manageViewVisibility();

        if (findViewById(R.id.otpTxnCancelLayoutId) != null)
            findViewById(R.id.otpTxnCancelLayoutId).setVisibility(View.GONE);

        if (findViewById(R.id.otpTxnSuccessfulLayoutId) != null)
            findViewById(R.id.otpTxnSuccessfulLayoutId).setVisibility(View.VISIBLE);

        if (paymentStatus == PaymentStatus.SUCCESS) {
            int resourceId = context.getResources().getIdentifier("img_checkmark_green", "drawable", context.getPackageName());

            Drawable drawable;
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                drawable = context.getResources().getDrawable(resourceId, null);
            } else {
                drawable = context.getResources().getDrawable(resourceId);
            }
            ((ImageView) findViewById(R.id.otpTxnDoneImgViewId)).setImageDrawable(drawable);

            findViewById(R.id.otpTxnDoneImgViewId).startAnimation(fadeIn);
            ((TypeWriterTextView) findViewById(R.id.otpTxnSuccessTxtId)).animateText("Transaction Successful", 20);
        } else if (paymentStatus == PaymentStatus.FAILED ||
                paymentStatus == PaymentStatus.CANCELLED ||
                paymentStatus == PaymentStatus.OTHERS) {

            int resourceId = context.getResources().getIdentifier("img_cross_red", "drawable", context.getPackageName());

            Drawable drawable;
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                drawable = context.getResources().getDrawable(resourceId, null);
            } else {
                drawable = context.getResources().getDrawable(resourceId);
            }
            ((ImageView) findViewById(R.id.otpTxnDoneImgViewId)).setImageDrawable(drawable);
            findViewById(R.id.otpTxnDoneImgViewId).startAnimation(fadeIn);
            String txnMessage = null;
            if (paymentStatus == PaymentStatus.CANCELLED) {
                txnMessage = "Transaction Cancelled";
            } else if (paymentStatus == PaymentStatus.FAILED) {
                txnMessage = "Transaction Failed";
            } else if (paymentStatus == PaymentStatus.OTHERS) {
                txnMessage = "Transaction Failed";
            }
            ((TypeWriterTextView) findViewById(R.id.otpTxnSuccessTxtId)).animateText(txnMessage, 20);
        }
    }

    /**
     * Display OTP detect OTP scenarios.
     */
    public void displayOtpAutoDetectPopup() {
        removeAllViewsInLayout();
        inflate(context, R.layout.otppopup_waiting, this);
        if (otpAutoReadState == OTPAutoReadState.AutoReadEnabled) {
            enterOtpYourself = false;
            handleAutoReadEnabledUI();
        } else if (otpAutoReadState == OTPAutoReadState.AutoReadDisabled) {
            enterOtpYourself = true;
            if (resendClicked) {
                listener.displayOTPErrorMessage("OTP Resent! SMS Read Permission not Granted. Enter OTP manually.");
            } else {
                listener.displayOTPErrorMessage("SMS Read Permission not Granted. Enter OTP manually.");
            }
            handleAutoReadDisableUI();
        }
    }

    /**
     * Onclick callbacks.
     *
     * @param view
     */
    @Override
    public void onClick(View view) {
        int i = view.getId();
        if (i == R.id.continuePasswordTextViewId) {
            displayEnterPasswordPopup();
            listener.onEnterPasswordClicked();
        }
        if (i == R.id.enterOTPYourselfTextViewId) {
            // Enter OTP yourself
            enterOtpYourself = true;
            listener.stopOtpReadTimer();
        }
        if (i == R.id.otpResendBtnId) {
            enterOtpYourself = false;
            KeyBoardUtil.hideKeyboard((Activity) context);
            handleResendOTPAction();
        }
    }


    @Override
    public void onComplete() {
        if (findViewById(R.id.otpWaitingLayoutId) != null) {
            findViewById(R.id.otpWaitingLayoutId).setVisibility(View.GONE);
        }
        if (progressGenerator.getProcessBtn() == btnRequestOTP) {
            handleRequestOTPAction();
        }
        if (progressGenerator.getProcessBtn() == otpConfirmBtn) {
            // Confirm OTP Button.
            otpConfirmBtn.setClickable(true);
        }
        if (progressGenerator.getProcessBtn() == btnSubmitPass) {
        }
    }

    /**
     * Request OTP scenario.
     */
    private void handleRequestOTPAction() {
        // Request OTP Button.
        btnRequestOTP.setClickable(true);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
//             Check for Android 6.0 +
            if (smsPermissionGranted) {
                // SMS permission is granted hence show OTP Auto Reading UI
                Log.d("SMSPermitted,autoread", "");
                otpAutoReadState = OTPAutoReadState.AutoReadEnabled;
            } else {
                // SMS permission is denied hence show otp manual entry UI
                Log.d("SMSnotPermitted,manual", "");
                enterOtpYourself = true;
                otpAutoReadState = OTPAutoReadState.AutoReadDisabled;
            }
        } else {
            // older Android versions
            otpAutoReadState = OTPAutoReadState.AutoReadEnabled;
        }
        displayOtpAutoDetectPopup();
    }

    /**
     * Resend OTP scenario.
     */
    private void handleResendOTPAction() {
        // Resend OTP Button.
        resendClicked = true;
        enterOtpEditTxt.clearComposingText();
        listener.onResendOTPClicked();
//        displayOtpAutoDetectPopup();
    }

    /**
     * OTP Auto READ Disabled UI.
     */
    private void handleAutoReadDisableUI() {

        removeAllViewsInLayout();
        inflate(context, R.layout.otp_autoread_timeout, this);

        this.otpResendBtn = (Button) findViewById(R.id.otpResendBtnId);
        this.otpResendBtn.setOnClickListener(this);

        this.otpConfirmBtn = (ActionProcessButton) findViewById(R.id.otpConfirmBtnId);
        this.otpConfirmBtn.setMode(ActionProcessButton.Mode.PROGRESS);
        this.otpConfirmBtn.setOnClickNormalState(new OnClickListener() {
            @Override
            public void onClick(View v) {
                KeyBoardUtil.hideKeyboard((Activity) context);
                userEnteredOtp = enterOtpEditTxt.getText().toString();
                if (TextUtils.isEmpty(userEnteredOtp)) {
                    // Otp detection failed or timeout and no otp entered
                    enterOtpEditTxt.requestFocus();
                    enterOtpEditTxt.setError("Please enter OTP or click Resend");
                } else {
                    // Otp detection failed or timeout and user entered it manually
                    progressGenerator.setRandomDelayRange(900);
                    progressGenerator.start(otpConfirmBtn);
                    otpConfirmBtn.setClickable(false);
                    listener.onProcessTransactionClicked(userEnteredOtp);
                }

            }
        }).build();

        this.enterOtpEditTxt = (EditText) findViewById(R.id.enterOtpEditTextId);
        this.enterOtpEditTxt.requestFocus();

        if (findViewById(R.id.otpTimeoutLayoutId) != null) {
            findViewById(R.id.otpTimeoutLayoutId).setVisibility(View.VISIBLE);
            this.otpResendBtn.setEnabled(true);
            this.otpResendBtn.setClickable(true);

            if (findViewById(R.id.otpWaitingLayoutId) != null)
                findViewById(R.id.otpWaitingLayoutId).setVisibility(View.INVISIBLE);

            // This happens if user chooses to enter otp manually,
            // or if SMS permission is denied.
            if (enterOtpYourself) {
                otpResendBtn.setEnabled(false);
                otpResendBtn.setClickable(false);

                startResendButtonVisibilityTimer();
            }
        }

        if (bankJsConfig.getHideResendBtn()) {
            this.otpResendBtn.setVisibility(View.GONE);
        }
    }

    private void startResendButtonVisibilityTimer() {
        if (!mResendButtonVisibilityTimerRunning) {
            mStartResendButtonVisibilityTimer.start();
        }
    }

    /**
     * OTP Auto READ Enabled UI.
     */
    private void handleAutoReadEnabledUI() {

        findViewById(R.id.otpWaitingLayoutId).setVisibility(View.VISIBLE);
        findViewById(R.id.enterOTPYourselfTextViewId).setOnClickListener(this);

        DilatingDotsProgressBar mDilatingDotsProgressBar = (DilatingDotsProgressBar) findViewById(R.id.progress);

        // show progress bar and start animating
        mDilatingDotsProgressBar.show(100);
    }

    /**
     * Card Unregistered scenario.
     */
    public void showCardUnregistered() {

        if (findViewById(R.id.otpPopupView1Id) != null)
            findViewById(R.id.otpPopupView1Id).setVisibility(View.VISIBLE);
        if (findViewById(R.id.otpRegisterCardTxtId) != null)
            findViewById(R.id.otpRegisterCardTxtId).setVisibility(View.VISIBLE);
        if (findViewById(R.id.requestOTPBtnId) != null)
            findViewById(R.id.requestOTPBtnId).setVisibility(View.GONE);
        if (findViewById(R.id.orTextId) != null)
            findViewById(R.id.orTextId).setVisibility(View.GONE);
        if (findViewById(R.id.continuePasswordTextViewId) != null)
            findViewById(R.id.continuePasswordTextViewId).setVisibility(View.GONE);
        if (findViewById(R.id.otpRegisterCardTxtId) != null)
            ((TypeWriterTextView) findViewById(R.id.otpRegisterCardTxtId)).animateText("Please register your Card!", 5);
        // Dismiss OTP Popup as card is unregistered.
        listener.dismissOTPPopup();

    }

    /**
     * Card Unsupported Scenario.
     */
    public void showCardUnSupported() {

        if (findViewById(R.id.otpPopupView1Id) != null)
            findViewById(R.id.otpPopupView1Id).setVisibility(View.VISIBLE);
        if (findViewById(R.id.otpRegisterCardTxtId) != null)
            findViewById(R.id.otpRegisterCardTxtId).setVisibility(View.VISIBLE);
        if (findViewById(R.id.requestOTPBtnId) != null)
            findViewById(R.id.requestOTPBtnId).setVisibility(View.GONE);
        if (findViewById(R.id.orTextId) != null)
            findViewById(R.id.orTextId).setVisibility(View.GONE);
        if (findViewById(R.id.continuePasswordTextViewId) != null)
            findViewById(R.id.continuePasswordTextViewId).setVisibility(View.GONE);
//        if (findViewById(R.id.otpRegisterCardTxtId) != null)
//            ((TypeWriterTextView) findViewById(R.id.otpRegisterCardTxtId)).animateText("Card unsupported currently!", 5);
        // Dismiss OTP Popup as card is unsupported.
        listener.dismissOTPPopup();
    }

    /**
     * OTP Auto READ State.
     *
     * @param otpAutoReadState
     */
    public void setOtpAutoReadState(OTPAutoReadState otpAutoReadState) {
        this.otpAutoReadState = otpAutoReadState;
    }

    /**
     * OTP set Listener.
     *
     * @param listener
     */
    public void setListener(OTPViewListener listener) {
        this.listener = listener;
    }
}
