package com.citruspay.citrusbrowser.listener;

/**
 * This class performs as a interface between the Citrus WebView and the Flash UI. Created by gautam
 * on 8/16/16.
 */

import android.content.Context;
import android.text.TextUtils;

public class Presenter {

    private static Presenter presenter;
    private final Context context;
    private FlashObserver flashObserver = null;
    private CitrusWebViewObserver citrusWebViewObserver = null;
    private FlashState state;

    /**
     * Flash UI States.
     */
    public enum FlashState {
        PAGE_TIMER_START,
        PAGE_TIMER_STOP, CARD_UNREGISTERED, PAYMENT_FAILED, LOAD_JS, AI_BANK_NAME, AI_CARD_SCHEME, AI_CARD_TYPE, CITI_CARD_VISIBILITY,
    }

    /**
     * Citrus WebView States.
     */
    public enum CitrusWebViewState {
        LOAD_URL,
        STOP_LOADING, HIDE_WEBVIEW, SHOW_WEBVIEW, GO_BACK,
        CHECK_CITI_OTP2_VIEW_VISIBILITY
    }

    private Presenter(Context ctx) {
        context = ctx;
    }

    /**
     * Get Presenter instance.
     *
     * @param context
     * @return
     */
    public static Presenter getInstance(Context context) {
        if (presenter == null) {
            synchronized (Presenter.class) {
                if (presenter == null) {
                    presenter = new Presenter(context);
                }
            }
        }
        return presenter;
    }

    /**
     * Register for Citrus Webview callback events.
     *
     * @param observer
     */
    public void listen(CitrusWebViewObserver observer) {
        this.citrusWebViewObserver = observer;
    }

    /**
     * Register for Flash callback events,
     *
     * @param observer
     */
    public void listen(FlashObserver observer) {
        this.flashObserver = observer;
    }

    /**
     * Notify the action to perform on Citrus Webview.
     *
     * @param state
     * @param url
     */
    public void notifyCitrusWebView(CitrusWebViewState state, String url) {
        if (citrusWebViewObserver != null) {
            switch (state) {
                case LOAD_URL:
                    citrusWebViewObserver.loadJS(url);
                    break;
                case STOP_LOADING:
                    citrusWebViewObserver.stopLoading();
                    break;
                case HIDE_WEBVIEW:
                    citrusWebViewObserver.hideWebView();
                    break;
                case SHOW_WEBVIEW:
                    citrusWebViewObserver.showWebView();
                    break;
                case GO_BACK:
                    citrusWebViewObserver.goBack();
                    break;
                case CHECK_CITI_OTP2_VIEW_VISIBILITY:
                    if (!TextUtils.isEmpty(url)) {
                        citrusWebViewObserver.loadJS(url);
                    }
                    break;
            }
        }
    }

    /**
     * Notify the action to perform on Flash
     *
     * @param state
     */
    public void notifyFlash(FlashState state, String msg) {
        if (flashObserver != null) {
            switch (state) {
                case PAGE_TIMER_START:
                    flashObserver.timerStart();
                    break;
                case PAGE_TIMER_STOP:
                    flashObserver.timerStop();
                    break;
                case CARD_UNREGISTERED:
                    flashObserver.cardUnregistered();
                    break;
                case PAYMENT_FAILED:
                    flashObserver.paymentFailed();
                    break;
                case CITI_CARD_VISIBILITY:
                    flashObserver.onReceivedCitiOTP2ViewVisibility(msg);
                    break;
            }
        }
    }

}