package com.citruspay.citrusbrowser.core;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.Fragment;
import android.content.Context;
import android.graphics.Bitmap;
import android.net.http.SslError;
import android.os.Build;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.util.Log;
import android.view.View;
import android.view.ViewGroup;
import android.webkit.CookieManager;
import android.webkit.JavascriptInterface;
import android.webkit.SslErrorHandler;
import android.webkit.WebSettings;
import android.webkit.WebView;
import android.webkit.WebViewClient;

import com.citruspay.citrusbrowser.listener.CitrusWebViewObserver;
import com.citruspay.citrusbrowser.listener.Presenter;
import com.citruspay.citrusbrowser.utils.Constants;

import java.lang.ref.WeakReference;
import java.util.HashMap;
import java.util.Map;

/**
 * Created by gautam on 8/16/16.
 */
public class CitrusWebView extends WebView implements CitrusWebViewObserver {

    private Context mContext = null;
    private Presenter presenter = null;
    private FlashListener mFlashListener = null;
    protected static final String CHARSET_DEFAULT = "UTF-8";
    protected WeakReference<Activity> mActivity;
    protected WeakReference<Fragment> mFragment;
    protected CitrusWebViewListener mCitrusWebViewListener;
    protected long mLastError;
    protected final Map<String, String> mHttpHeaders = new HashMap<String, String>();

    /**
     * Callback for CitrusWebView events.
     */
    public interface CitrusWebViewListener {
        void onPageStarted(String url, Bitmap favicon);

        void onPageFinished(String url);

        void onPageError(int errorCode, String description, String failedUrl);

        void shouldOverrideUrlLoading(WebView view, String url);
    }

    /**
     * Callback for flash events.
     */
    public interface FlashListener {
        /**
         * This is the callback event to indicate that the
         * flash should be started.
         *
         * @param flashResponseData
         */
        void startFlash(FlashResponseData flashResponseData);
    }

    public CitrusWebView(Context context) {
        super(context);
        this.mContext = context;
        init(context);
    }

    public CitrusWebView(Context context, AttributeSet attrs) {
        super(context, attrs);
        this.mContext = context;
        init(context);
    }

    public CitrusWebView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        this.mContext = context;
        init(context);
    }

    public void setFlashListener(final FlashListener listener) {
        mFlashListener = listener;
    }

    public void setCitrusWebViewListener(final Activity activity, final CitrusWebViewListener citrusWebViewListener) {
        if (activity != null) {
            mActivity = new WeakReference<>(activity);
        } else {
            mActivity = null;
        }

        mCitrusWebViewListener = citrusWebViewListener;
    }

    public void setCitrusWebViewListener(final Fragment fragment, final CitrusWebViewListener citrusWebViewListener) {
        if (fragment != null) {
            mFragment = new WeakReference<>(fragment);
        } else {
            mFragment = null;
        }

        mCitrusWebViewListener = citrusWebViewListener;
    }


    /**
     * Loads and displays the provided HTML source text
     *
     * @param html the HTML source text to load
     */
    public void loadHtml(final String html) {
        loadHtml(html, null);
    }

    /**
     * Loads and displays the provided HTML source text
     *
     * @param html    the HTML source text to load
     * @param baseUrl the URL to use as the page's base URL
     */
    public void loadHtml(final String html, final String baseUrl) {
        loadHtml(html, baseUrl, null);
    }

    /**
     * Loads and displays the provided HTML source text
     *
     * @param html       the HTML source text to load
     * @param baseUrl    the URL to use as the page's base URL
     * @param historyUrl the URL to use for the page's history entry
     */
    public void loadHtml(final String html, final String baseUrl, final String historyUrl) {
        loadHtml(html, baseUrl, historyUrl, "utf-8");
    }

    /**
     * Loads and displays the provided HTML source text
     *
     * @param html       the HTML source text to load
     * @param baseUrl    the URL to use as the page's base URL
     * @param historyUrl the URL to use for the page's history entry
     * @param encoding   the encoding or charset of the HTML source text
     */
    public void loadHtml(final String html, final String baseUrl, final String historyUrl, final String encoding) {
        loadDataWithBaseURL(baseUrl, html, "text/html", encoding, historyUrl);
    }

    @SuppressLint("NewApi")
    @SuppressWarnings("all")
    public void onResume() {
        if (Build.VERSION.SDK_INT >= 11) {
            super.onResume();
        }
        resumeTimers();
    }

    @SuppressLint("NewApi")
    @SuppressWarnings("all")
    public void onPause() {
        pauseTimers();
        if (Build.VERSION.SDK_INT >= 11) {
            super.onPause();
        }
    }

    public void onDestroy() {
        // try to remove this view from its parent first
        try {
            ((ViewGroup) getParent()).removeView(this);
        } catch (Exception ignored) {
        }

        // then try to remove all child views from this view
        try {
            removeAllViews();
        } catch (Exception ignored) {
        }

        // and finally destroy this view
        destroy();
    }

    /**
     * Adds an additional HTTP header that will be sent along with every HTTP `GET` request
     * <p>
     * This does only affect the main requests, not the requests to included resources (e.g. images)
     * <p>
     * If you later want to delete an HTTP header that was previously added this way, call `removeHttpHeader()`
     * <p>
     * The `WebView` implementation may in some cases overwrite headers that you set or unset
     *
     * @param name  the name of the HTTP header to add
     * @param value the value of the HTTP header to send
     */
    public void addHttpHeader(final String name, final String value) {
        mHttpHeaders.put(name, value);
    }

    /**
     * Removes one of the HTTP headers that have previously been added via `addHttpHeader()`
     * <p>
     * If you want to unset a pre-defined header, set it to an empty string with `addHttpHeader()` instead
     * <p>
     * The `WebView` implementation may in some cases overwrite headers that you set or unset
     *
     * @param name the name of the HTTP header to remove
     */
    public void removeHttpHeader(final String name) {
        mHttpHeaders.remove(name);
    }

    public void onBackPressed() {
        if (canGoBack()) {
            goBack();
        }
    }

    public void setCookiesEnabled(final boolean enabled) {
        CookieManager.getInstance().setAcceptCookie(enabled);
    }

    @SuppressLint("NewApi")
    public void setThirdPartyCookiesEnabled(final boolean enabled) {
        if (Build.VERSION.SDK_INT >= 21) {
            CookieManager.getInstance().setAcceptThirdPartyCookies(this, enabled);
        }
    }

    @SuppressWarnings("static-method")
    @SuppressLint("NewApi")
    protected void setMixedContentAllowed(final WebSettings webSettings, final boolean allowed) {
        if (Build.VERSION.SDK_INT >= 21) {
            webSettings.setMixedContentMode(allowed ? WebSettings.MIXED_CONTENT_ALWAYS_ALLOW : WebSettings.MIXED_CONTENT_NEVER_ALLOW);
        }
    }

    @SuppressLint({"SetJavaScriptEnabled", "AddJavascriptInterface"})
    protected void init(Context context) {
        // in IDE's preview mode
        if (isInEditMode()) {
            // do not run the code from this method
            return;
        }

        if (context instanceof Activity) {
            mActivity = new WeakReference<Activity>((Activity) context);
        }

        presenter = Presenter.getInstance(mContext);
        presenter.listen(this);

        setFocusable(true);
        setFocusableInTouchMode(true);

        setSaveEnabled(true);

        super.addJavascriptInterface(new CitrusJSInterface(), Constants.FLASH_JS_INTERFACE_NAME);

        final WebSettings webSettings = getSettings();
        webSettings.setAllowFileAccess(false);
        webSettings.setBuiltInZoomControls(false);
        webSettings.setJavaScriptEnabled(true);
        webSettings.setDomStorageEnabled(true);
        if (Build.VERSION.SDK_INT < 18) {
            webSettings.setRenderPriority(WebSettings.RenderPriority.HIGH);
        }
        setMixedContentAllowed(webSettings, true);

        setThirdPartyCookiesEnabled(true);

        super.setWebViewClient(new CitrusWebViewClient());

    }

    @Override
    public void loadUrl(final String url, Map<String, String> additionalHttpHeaders) {
        if (additionalHttpHeaders == null) {
            additionalHttpHeaders = mHttpHeaders;
        } else if (mHttpHeaders.size() > 0) {
            additionalHttpHeaders.putAll(mHttpHeaders);
        }

        if (!TextUtils.isEmpty(url) && additionalHttpHeaders != null)
            super.loadUrl(url, additionalHttpHeaders);
    }


    @Override
    public void loadUrl(final String url) {
        if (mHttpHeaders.size() > 0) {
            if (!TextUtils.isEmpty(url))
                super.loadUrl(url, mHttpHeaders);
        } else {
            if (!TextUtils.isEmpty(url))
                super.loadUrl(url);
        }
    }

    protected void setLastError() {
        mLastError = System.currentTimeMillis();
    }

    protected boolean hasError() {
        return (mLastError + 500) >= System.currentTimeMillis();
    }


    /**
     * This class will be loaded as JSInterface and the methods of this class will be called from
     * the javascript loaded inside webview.
     * <p>
     * Handle the payment response and take actions accordingly.
     */
    private class CitrusJSInterface {

        @JavascriptInterface
        public void cardData(String response) {

            Log.d("CARD DATA :: ", response);
            FlashResponseData jsInterfaceResponse = new FlashResponseData();
            jsInterfaceResponse.parseResponse(response);
            if (jsInterfaceResponse.getTxnDetails().getTxnType() == TxnType.CREDIT_CARD || jsInterfaceResponse.getTxnDetails().getTxnType() == TxnType.DEBIT_CARD) {
                startFlash(jsInterfaceResponse);
            }
        }

        @JavascriptInterface
        public void printHTML(String response) {
            if (!TextUtils.isEmpty(response)) {

                if (response.contains("Un-Registered") /*|| response.contains("Registration") */ || response.contains("Step 1")) {
                    notifyFlash(Presenter.FlashState.CARD_UNREGISTERED, null);
                }
            }
        }
    }

    private void startFlash(FlashResponseData flashResponseData) {
        if (mFlashListener != null)
            mFlashListener.startFlash(flashResponseData);
    }

    private void notifyFlash(Presenter.FlashState state, String msg) {
        if (presenter != null)
            presenter.notifyFlash(state, msg);
    }


    /**
     * Handle all the Webview loading in custom webview client.
     */
    private class CitrusWebViewClient extends WebViewClient {

        @Override
        public void onPageStarted(WebView view, String url, Bitmap favicon) {
            if (!hasError()) {
                if (mCitrusWebViewListener != null) {
                    mCitrusWebViewListener.onPageStarted(url, favicon);
                }
            }
            notifyFlash(Presenter.FlashState.PAGE_TIMER_STOP, null);
        }

        @Override
        public void onPageFinished(WebView view, String url) {
            if (!hasError()) {
                if (mCitrusWebViewListener != null) {
                    mCitrusWebViewListener.onPageFinished(url);
                }
            }
            view.loadUrl("javascript:window.CitrusFlashPaymentStartedAndroid.printHTML('<html>'+document.getElementsByTagName('html')[0].innerHTML+'</html>');");

            notifyFlash(Presenter.FlashState.PAGE_TIMER_START, null);
        }

        @Override
        public void onReceivedError(WebView view, int errorCode, String description, String failingUrl) {
            setLastError();

            if (mCitrusWebViewListener != null) {
                mCitrusWebViewListener.onPageError(errorCode, description, failingUrl);
            }
        }

        @Override
        public boolean shouldOverrideUrlLoading(final WebView view, final String url) {

            // Txn has failed.
            if (url.contains("vpc_TxnResponseCode=F")) {
                notifyFlash(Presenter.FlashState.PAYMENT_FAILED, null);
            }

            if (mCitrusWebViewListener != null) {
                mCitrusWebViewListener.shouldOverrideUrlLoading(view, url);
            }

            // route the request through the custom URL loading method
            view.loadUrl(url);

            // cancel the original request
            return true;
        }

        @Override
        public void onReceivedSslError(WebView view, SslErrorHandler handler, SslError error) {
            super.onReceivedSslError(view, handler, error);
            // Cancelling loading of the page.
            handler.cancel();
        }
    }

    @Override
    public void loadJS(String js) {
        super.loadUrl(js);
    }

    @Override
    public void stopLoading() {
        super.stopLoading();
    }

    @Override
    public void hideWebView() {
        setVisibility(View.GONE);
    }

    @Override
    public void showWebView() {
        setVisibility(View.VISIBLE);
    }

}
