/*
 * Decompiled with CFR 0.152.
 */
package com.chutneytesting.execution.domain.scenario;

import com.chutneytesting.design.domain.dataset.DataSetHistoryRepository;
import com.chutneytesting.design.domain.scenario.TestCase;
import com.chutneytesting.execution.domain.ExecutionRequest;
import com.chutneytesting.execution.domain.compiler.TestCasePreProcessors;
import com.chutneytesting.execution.domain.history.ExecutionHistory;
import com.chutneytesting.execution.domain.history.ExecutionHistoryRepository;
import com.chutneytesting.execution.domain.history.ImmutableExecutionHistory;
import com.chutneytesting.execution.domain.report.ScenarioExecutionReport;
import com.chutneytesting.execution.domain.report.ServerReportStatus;
import com.chutneytesting.execution.domain.report.StepExecutionReportCore;
import com.chutneytesting.execution.domain.scenario.FailedExecutionAttempt;
import com.chutneytesting.execution.domain.scenario.ScenarioNotRunningException;
import com.chutneytesting.execution.domain.scenario.ServerTestEngine;
import com.chutneytesting.execution.domain.state.ExecutionStateRepository;
import com.chutneytesting.instrument.domain.ChutneyMetrics;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Ascii;
import com.google.common.base.Joiner;
import io.reactivex.Completable;
import io.reactivex.Observable;
import io.reactivex.schedulers.Schedulers;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ScenarioExecutionEngineAsync {
    private static final Logger LOGGER = LoggerFactory.getLogger(ScenarioExecutionEngineAsync.class);
    private static final long DEFAULT_RETENTION_DELAY_SECONDS = 5L;
    private static final long DEFAULT_DEBOUNCE_MILLISECONDS = 100L;
    private final ObjectMapper objectMapper;
    private final ExecutionHistoryRepository executionHistoryRepository;
    private final ServerTestEngine executionEngine;
    private final ExecutionStateRepository executionStateRepository;
    private final ChutneyMetrics metrics;
    private final TestCasePreProcessors testCasePreProcessors;
    private final DataSetHistoryRepository dataSetHistoryRepository;
    private Map<Long, Pair<Observable<ScenarioExecutionReport>, Long>> scenarioExecutions = new ConcurrentHashMap<Long, Pair<Observable<ScenarioExecutionReport>, Long>>();
    private long retentionDelaySeconds;
    private long debounceMilliSeconds;

    public ScenarioExecutionEngineAsync(ExecutionHistoryRepository executionHistoryRepository, ServerTestEngine executionEngine, ExecutionStateRepository executionStateRepository, ChutneyMetrics metrics, TestCasePreProcessors testCasePreProcessors, ObjectMapper objectMapper, DataSetHistoryRepository dataSetHistoryRepository) {
        this(executionHistoryRepository, executionEngine, executionStateRepository, metrics, testCasePreProcessors, objectMapper, dataSetHistoryRepository, 5L, 100L);
    }

    public ScenarioExecutionEngineAsync(ExecutionHistoryRepository executionHistoryRepository, ServerTestEngine executionEngine, ExecutionStateRepository executionStateRepository, ChutneyMetrics metrics, TestCasePreProcessors testCasePreProcessors, ObjectMapper objectMapper, DataSetHistoryRepository dataSetHistoryRepository, long retentionDelaySeconds, long debounceMilliSeconds) {
        this.executionHistoryRepository = executionHistoryRepository;
        this.executionEngine = executionEngine;
        this.executionStateRepository = executionStateRepository;
        this.metrics = metrics;
        this.testCasePreProcessors = testCasePreProcessors;
        this.objectMapper = objectMapper;
        this.dataSetHistoryRepository = dataSetHistoryRepository;
        this.retentionDelaySeconds = retentionDelaySeconds;
        this.debounceMilliSeconds = debounceMilliSeconds;
    }

    public Long execute(ExecutionRequest executionRequest) {
        ExecutionRequest executionRequestProcessed = new ExecutionRequest((TestCase)this.testCasePreProcessors.apply(executionRequest), executionRequest.environment, executionRequest.withExternalDataset, executionRequest.userId);
        ExecutionHistory.Execution storedExecution = this.storeInitialReport(executionRequestProcessed);
        Pair<Observable<StepExecutionReportCore>, Long> followResult = this.callEngineExecution(executionRequestProcessed, storedExecution);
        Observable<ScenarioExecutionReport> executionObservable = this.buildScenarioExecutionReportObservable(executionRequestProcessed, storedExecution.executionId(), followResult);
        LOGGER.trace("Add replayer for execution {}", (Object)storedExecution.executionId());
        this.scenarioExecutions.put(storedExecution.executionId(), (Pair<Observable<ScenarioExecutionReport>, Long>)Pair.of(executionObservable, (Object)((Long)followResult.getRight())));
        LOGGER.debug("Replayers map size : {}", (Object)this.scenarioExecutions.size());
        executionObservable.subscribeOn(Schedulers.io()).subscribe();
        return storedExecution.executionId();
    }

    private Pair<Observable<StepExecutionReportCore>, Long> callEngineExecution(ExecutionRequest executionRequest, ExecutionHistory.Execution storedExecution) {
        Pair<Observable<StepExecutionReportCore>, Long> followResult;
        try {
            followResult = this.executionEngine.executeAndFollow(executionRequest);
        }
        catch (Exception e) {
            LOGGER.error("Cannot execute test case [" + executionRequest.testCase.id() + "]", (Object)e.getMessage());
            this.setExecutionToFailed(executionRequest.testCase.id(), storedExecution, Optional.ofNullable(e.getMessage()).orElse(e.toString()));
            throw new FailedExecutionAttempt(e, storedExecution.executionId(), executionRequest.testCase.metadata().title());
        }
        return followResult;
    }

    Observable<ScenarioExecutionReport> buildScenarioExecutionReportObservable(ExecutionRequest executionRequest, Long executionId, Pair<Observable<StepExecutionReportCore>, Long> engineExecution) {
        Observable replayer = ((Observable)engineExecution.getLeft()).observeOn(Schedulers.io());
        if (this.debounceMilliSeconds > 0L) {
            replayer = replayer.debounce(this.debounceMilliSeconds, TimeUnit.MILLISECONDS);
        }
        return replayer.doOnSubscribe(disposable -> this.notifyExecutionStart(executionId, executionRequest.testCase)).map(report -> {
            LOGGER.trace("Map report for execution {}", (Object)executionId);
            return new ScenarioExecutionReport(executionId, executionRequest.testCase.metadata().title(), executionRequest.environment, executionRequest.userId, (StepExecutionReportCore)report);
        }).doOnNext(report -> this.updateHistory(executionId, executionRequest, (ScenarioExecutionReport)report)).doOnTerminate(() -> this.notifyExecutionEnd(executionId, executionRequest.testCase)).doOnTerminate(() -> this.sendMetrics(executionId, executionRequest.testCase)).doOnTerminate(() -> this.cleanExecutionId(executionId)).replay(1).autoConnect();
    }

    private void setExecutionToFailed(String scenarioId, ExecutionHistory.Execution storedExecution, String errorMessage) {
        ImmutableExecutionHistory.Execution execution = ImmutableExecutionHistory.Execution.copyOf(storedExecution).withStatus(ServerReportStatus.FAILURE).withError(errorMessage);
        this.executionHistoryRepository.update(scenarioId, execution);
    }

    private ExecutionHistory.Execution storeInitialReport(ExecutionRequest executionRequest) {
        Optional<Pair<String, Integer>> executionDataSet = this.findExecutionDataset(executionRequest);
        ImmutableExecutionHistory.DetachedExecution detachedExecution = ImmutableExecutionHistory.DetachedExecution.builder().time(LocalDateTime.now()).duration(0L).status(ServerReportStatus.RUNNING).info("").error("").report("").testCaseTitle(executionRequest.testCase.metadata().title()).environment(executionRequest.environment).datasetId(executionDataSet.map(Pair::getLeft)).datasetVersion(executionDataSet.map(Pair::getRight)).user(executionRequest.userId).build();
        return this.executionHistoryRepository.store(executionRequest.testCase.id(), detachedExecution);
    }

    public Observable<ScenarioExecutionReport> followExecution(String scenarioId, Long executionId) {
        if (this.scenarioExecutions.containsKey(executionId)) {
            return (Observable)this.scenarioExecutions.get(executionId).getLeft();
        }
        throw new ScenarioNotRunningException(scenarioId);
    }

    public void stop(String scenarioId, Long executionId) {
        if (!this.scenarioExecutions.containsKey(executionId)) {
            throw new ScenarioNotRunningException(scenarioId);
        }
        this.executionEngine.stop((Long)this.scenarioExecutions.get(executionId).getRight());
    }

    public void pause(String scenarioId, Long executionId) {
        if (!this.scenarioExecutions.containsKey(executionId)) {
            throw new ScenarioNotRunningException(scenarioId);
        }
        this.executionEngine.pause((Long)this.scenarioExecutions.get(executionId).getRight());
    }

    public void resume(String scenarioId, Long executionId) {
        if (!this.scenarioExecutions.containsKey(executionId)) {
            throw new ScenarioNotRunningException(scenarioId);
        }
        this.executionEngine.resume((Long)this.scenarioExecutions.get(executionId).getRight());
    }

    public void setRetentionDelaySeconds(long retentionDelaySeconds) {
        this.retentionDelaySeconds = retentionDelaySeconds;
    }

    public void setDebounceMilliSeconds(long debounceMilliSeconds) {
        this.debounceMilliSeconds = debounceMilliSeconds;
    }

    private ExecutionHistory.DetachedExecution summarize(ScenarioExecutionReport scenarioReport, String environment, String userId) {
        return ImmutableExecutionHistory.DetachedExecution.builder().time(scenarioReport.report.startDate.atZone(ZoneId.systemDefault()).toLocalDateTime()).duration(scenarioReport.report.duration).status(scenarioReport.report.status).info(this.joinAndTruncateMessages(ScenarioExecutionEngineAsync.searchInfo(scenarioReport.report))).error(this.joinAndTruncateMessages(ScenarioExecutionEngineAsync.searchErrors(scenarioReport.report))).report(this.serialize(scenarioReport)).testCaseTitle(scenarioReport.scenarioName).environment(environment).user(userId).build();
    }

    private String serialize(ScenarioExecutionReport stepExecutionReport) {
        try {
            return this.objectMapper.writeValueAsString((Object)stepExecutionReport);
        }
        catch (JsonProcessingException e) {
            LOGGER.error("Unable to serialize StepExecutionReport content with name='{}'", (Object)stepExecutionReport.report.name, (Object)e);
            return "{}";
        }
    }

    private Optional<String> joinAndTruncateMessages(Iterable<String> messages) {
        return Optional.of(Ascii.truncate((CharSequence)Joiner.on((String)", ").join(messages), (int)50, (String)"...")).filter(s -> !s.isEmpty());
    }

    private void notifyExecutionStart(long executionId, TestCase testCase) {
        LOGGER.trace("Notify start for execution {}", (Object)executionId);
        this.executionStateRepository.notifyExecutionStart(testCase.id());
    }

    private void cleanExecutionId(long executionId) {
        LOGGER.trace("Clean for execution {}", (Object)executionId);
        if (this.retentionDelaySeconds > 0L) {
            Completable.timer((long)this.retentionDelaySeconds, (TimeUnit)TimeUnit.SECONDS).subscribe(() -> {
                LOGGER.trace("Remove replayer for execution {}", (Object)executionId);
                this.scenarioExecutions.remove(executionId);
            }, throwable -> LOGGER.error("Cannot remove replayer for execution {}", (Object)executionId, throwable));
        } else {
            this.scenarioExecutions.remove(executionId);
        }
    }

    private void sendMetrics(long executionId, TestCase testCase) {
        LOGGER.trace("Send metrics for execution {}", (Object)executionId);
        try {
            ExecutionHistory.Execution execution = this.executionHistoryRepository.getExecution(testCase.id(), executionId);
            this.metrics.onScenarioExecutionEnded(testCase, execution);
        }
        catch (Exception e) {
            LOGGER.error("Send metrics for execution {} failed", (Object)executionId, (Object)e);
        }
    }

    private void updateHistory(long executionId, ExecutionRequest executionRequest, ScenarioExecutionReport report) {
        LOGGER.trace("Update history for execution {}", (Object)executionId);
        try {
            this.executionHistoryRepository.update(executionRequest.testCase.id(), this.summarize(report, executionRequest.environment, executionRequest.userId).attach(executionId));
        }
        catch (Exception e) {
            LOGGER.error("Update history for execution {} failed", (Object)executionId, (Object)e);
        }
    }

    private void notifyExecutionEnd(long executionId, TestCase testCase) {
        LOGGER.trace("Notify end for execution {}", (Object)executionId);
        this.executionStateRepository.notifyExecutionEnd(testCase.id());
    }

    private static List<String> searchInfo(StepExecutionReportCore report) {
        if (report.information.isEmpty()) {
            return report.steps.stream().map(ScenarioExecutionEngineAsync::searchInfo).flatMap(Collection::stream).collect(Collectors.toList());
        }
        return report.information;
    }

    private static List<String> searchErrors(StepExecutionReportCore report) {
        if (report.errors.isEmpty()) {
            return report.steps.stream().map(ScenarioExecutionEngineAsync::searchErrors).flatMap(Collection::stream).collect(Collectors.toList());
        }
        return report.errors;
    }

    private Optional<Pair<String, Integer>> findExecutionDataset(ExecutionRequest executionRequest) {
        if (executionRequest.withExternalDataset) {
            return executionRequest.testCase.metadata().datasetId().map(datasetId -> Pair.of((Object)datasetId, (Object)this.dataSetHistoryRepository.lastVersion((String)datasetId)));
        }
        return Optional.empty();
    }
}

