/*
 * Decompiled with CFR 0.152.
 */
package com.chutneytesting.environment.domain;

import com.chutneytesting.environment.domain.Environment;
import com.chutneytesting.environment.domain.EnvironmentRepository;
import com.chutneytesting.environment.domain.EnvironmentVariable;
import com.chutneytesting.environment.domain.Target;
import com.chutneytesting.environment.domain.TargetFilter;
import com.chutneytesting.environment.domain.exception.AlreadyExistingEnvironmentException;
import com.chutneytesting.environment.domain.exception.AlreadyExistingTargetException;
import com.chutneytesting.environment.domain.exception.CannotDeleteEnvironmentException;
import com.chutneytesting.environment.domain.exception.EnvVariableNotFoundException;
import com.chutneytesting.environment.domain.exception.EnvironmentNotFoundException;
import com.chutneytesting.environment.domain.exception.InvalidEnvironmentNameException;
import com.chutneytesting.environment.domain.exception.NoEnvironmentFoundException;
import com.chutneytesting.environment.domain.exception.TargetNotFoundException;
import com.chutneytesting.environment.domain.exception.UnresolvedEnvironmentException;
import com.chutneytesting.environment.domain.exception.VariableAlreadyExistingException;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EnvironmentService {
    private static final String NAME_VALIDATION_REGEX = "[a-zA-Z0-9_\\-]{3,20}";
    private static final Pattern NAME_VALIDATION_PATTERN = Pattern.compile("[a-zA-Z0-9_\\-]{3,20}");
    private final Logger logger = LoggerFactory.getLogger(EnvironmentService.class);
    private final EnvironmentRepository environmentRepository;

    public EnvironmentService(EnvironmentRepository environmentRepository) {
        this.environmentRepository = environmentRepository;
    }

    public Set<String> listEnvironmentsNames() {
        return new LinkedHashSet<String>(this.environmentRepository.listNames());
    }

    public Set<Environment> listEnvironments() {
        return this.environmentRepository.listNames().stream().map(this.environmentRepository::findByName).collect(Collectors.toSet());
    }

    public Environment createEnvironment(Environment environment) throws InvalidEnvironmentNameException, AlreadyExistingEnvironmentException {
        return this.createEnvironment(environment, false);
    }

    public Environment createEnvironment(Environment environment, boolean force) throws InvalidEnvironmentNameException, AlreadyExistingEnvironmentException {
        if (!force && this.envAlreadyExist(environment)) {
            throw new AlreadyExistingEnvironmentException("Environment [" + environment.name + "] already exists");
        }
        this.createOrUpdate(environment);
        return environment;
    }

    public Environment getEnvironment(String environmentName) throws EnvironmentNotFoundException {
        return this.environmentRepository.findByName(environmentName);
    }

    public void deleteEnvironment(String environmentName) throws EnvironmentNotFoundException, CannotDeleteEnvironmentException {
        this.environmentRepository.delete(environmentName);
    }

    public void updateEnvironment(String environmentName, Environment newVersion) throws InvalidEnvironmentNameException, EnvironmentNotFoundException {
        Environment previousEnvironment = this.environmentRepository.findByName(environmentName);
        Environment newEnvironment = Environment.builder().from(previousEnvironment).withName(newVersion.name).withDescription(newVersion.description).build();
        this.createOrUpdate(newEnvironment);
        if (!newEnvironment.name.equals(environmentName)) {
            this.environmentRepository.delete(environmentName);
        }
    }

    public String defaultEnvironmentName() throws EnvironmentNotFoundException {
        List<String> envs = this.environmentRepository.listNames();
        if (envs.size() > 1) {
            throw new UnresolvedEnvironmentException("There is more than one environment. Could not resolve the default one");
        }
        if (envs.isEmpty()) {
            throw new NoEnvironmentFoundException("No Environment found");
        }
        return envs.get(0);
    }

    public List<Target> listTargets(TargetFilter filters) {
        Set<Object> targets = filters != null && StringUtils.isNotBlank((CharSequence)filters.environment()) ? this.environmentRepository.findByName((String)filters.environment()).targets : this.listEnvironments().stream().flatMap(environment -> environment.targets.stream()).collect(Collectors.toSet());
        return targets.stream().filter(target -> this.match((Target)target, filters)).collect(Collectors.toList());
    }

    public Set<String> listTargetsNames() {
        return this.listEnvironments().stream().flatMap(environment -> environment.targets.stream().map(target -> target.name)).collect(Collectors.toSet());
    }

    public Target getTarget(String environmentName, String targetName) {
        Environment environment = this.environmentRepository.findByName(environmentName);
        return environment.getTarget(targetName);
    }

    public void addTarget(Target target) throws EnvironmentNotFoundException, AlreadyExistingTargetException {
        Environment environment = this.environmentRepository.findByName(target.environment);
        Environment newEnvironment = environment.addTarget(target);
        this.createOrUpdate(newEnvironment);
    }

    public void deleteTarget(String environmentName, String targetName) throws EnvironmentNotFoundException, TargetNotFoundException {
        Environment environment = this.environmentRepository.findByName(environmentName);
        Environment newEnvironment = environment.deleteTarget(targetName);
        this.createOrUpdate(newEnvironment);
    }

    public void deleteTarget(String targetName) throws EnvironmentNotFoundException, TargetNotFoundException {
        this.environmentRepository.getEnvironments().stream().filter(env -> env.targets.stream().map(target -> target.name).toList().contains(targetName)).forEach(env -> {
            Environment newEnvironment = env.deleteTarget(targetName);
            this.createOrUpdate(newEnvironment);
        });
    }

    public void updateTarget(String previousTargetName, Target targetToUpdate) throws EnvironmentNotFoundException, TargetNotFoundException {
        Environment environment = this.environmentRepository.findByName(targetToUpdate.environment);
        Environment newEnvironment = environment.updateTarget(previousTargetName, targetToUpdate);
        this.createOrUpdate(newEnvironment);
        this.logger.debug("Updated target " + previousTargetName + " as " + targetToUpdate.name);
    }

    public void addVariable(List<EnvironmentVariable> values) throws EnvironmentNotFoundException, VariableAlreadyExistingException {
        values.forEach(variable -> {
            Environment environment = this.environmentRepository.findByName(variable.env());
            this.addVariable((EnvironmentVariable)variable, environment);
        });
    }

    public void createOrUpdateVariable(String existingKey, List<EnvironmentVariable> values) throws EnvironmentNotFoundException, EnvVariableNotFoundException {
        values.forEach(variable -> {
            Environment environment = this.environmentRepository.findByName(variable.env());
            if (!environment.containsVariable(existingKey)) {
                this.addVariable((EnvironmentVariable)variable, environment);
                return;
            }
            Environment updated = environment.updateVariable(existingKey, (EnvironmentVariable)variable);
            if (!environment.equals(updated)) {
                this.createOrUpdate(updated);
                this.logger.debug("Updated variable " + existingKey + " as " + ((EnvironmentVariable)values.get(0)).key());
            }
        });
    }

    public void deleteVariable(String key) {
        this.deleteVariable(key, this.environmentRepository.listNames());
    }

    public void deleteVariable(String key, List<String> envs) {
        List<Environment> environments = this.environmentRepository.findByNames(envs).stream().filter(env -> env.containsVariable(key)).toList();
        if (!envs.isEmpty() && environments.isEmpty()) {
            throw new EnvVariableNotFoundException("Variable [" + key + "] not found");
        }
        environments.forEach(env -> {
            Environment updated = env.deleteVariable(key);
            this.createOrUpdate(updated);
        });
        this.logger.debug("Deleted variable: " + key);
    }

    private void addVariable(EnvironmentVariable variable, Environment env) throws EnvironmentNotFoundException, VariableAlreadyExistingException {
        Environment updated = env.addVariable(variable);
        this.createOrUpdate(updated);
        this.logger.debug("Variable " + variable.key() + " added to environment " + env);
    }

    private void createOrUpdate(Environment environment) {
        if (!NAME_VALIDATION_PATTERN.matcher(environment.name).matches()) {
            throw new InvalidEnvironmentNameException("Environment name must be of 3 to 20 letters, digits, underscore or hyphen");
        }
        this.environmentRepository.save(environment);
    }

    private boolean envAlreadyExist(Environment environment) {
        return this.environmentRepository.listNames().stream().map(String::toUpperCase).toList().contains(environment.name.toUpperCase());
    }

    private boolean match(Target target, TargetFilter filters) {
        if (filters == null) {
            return true;
        }
        boolean matchName = true;
        if (StringUtils.isNotBlank((CharSequence)filters.name())) {
            matchName = filters.name().equals(target.name);
        }
        return matchName;
    }
}

