package com.atlassian.vcache.internal;

import java.util.Set;

/**
 * The lifecycle management interface that contains methods that need to be called, by the
 * host application, at different stages.
 * <p>
 * Handlers need to be provided that are called whenever a cache is about to be configured by the VCache
 * implementation. Using these hooks, the host application can:
 * </p>
 * <ul>
 * <li>Override the current cache settings. For example, enforcing an application wide maximum timeout.</li>
 * <li>
 * Log a warning (or whatever) based on it's policy (e.g. an unknown {@link com.atlassian.vcache.JvmCache}
 * is being created.
 * </li>
 * <li>
 * Prevent a cache from being created, based on it's policy, by throwing a {@link RuntimeException}. The
 * VCache implementation will propagate these exceptions to the original caller.
 * </li>
 * </ul>
 *
 * @since 1.0
 */
public interface VCacheLifecycleManager {
    /**
     * Synchronise the {@link com.atlassian.vcache.TransactionalExternalCache}'s that are associated with the
     * supplied context.
     *
     * @param context the request context to synchronise.
     */
    void transactionSync(RequestContext context);

    /**
     * Discard the {@link com.atlassian.vcache.TransactionalExternalCache}'s that are associated with the
     * supplied context.
     *
     * @param context the request context to discard.
     * @return names of the caches that have operations being discarded.
     */
    Set<String> transactionDiscard(RequestContext context);

    /**
     * Return the cache metrics for the supplied context.
     *
     * @param context the request context to obtain metrics for.
     * @return the cache metrics for the request
     */
    RequestMetrics metrics(RequestContext context);
}
