package com.atlassian.vcache.marshallers;

import com.atlassian.annotations.PublicApi;
import com.atlassian.vcache.Marshaller;

import javax.annotation.Nonnull;
import java.io.ObjectStreamClass;
import java.io.Serializable;
import java.util.Optional;

/**
 * Factory for {@link Marshaller} instances for common types.
 *
 * @since 1.0
 */
@PublicApi
public class MarshallerFactory {
    /**
     * Returns a {@link Marshaller} for {@link Optional} objects.
     *
     * @param valueMarshaller used to marshall values held by {@link Optional} instances.
     * @param <T>             the value type
     * @return a {@link Marshaller} for {@link Optional} objects.
     */
    @Nonnull
    public static <T> Marshaller<Optional<T>> optionalMarshaller(Marshaller<T> valueMarshaller) {
        return new OptionalMarshaller<>(valueMarshaller);
    }

    /**
     * Returns a {@link Marshaller} for {@link String} objects.
     *
     * @return a {@link Marshaller} for {@link String} objects.
     */
    @Nonnull
    public static Marshaller<String> stringMarshaller() {
        return new StringMarshaller();
    }

    /**
     * Returns a {@link Marshaller} for {@link Serializable} objects. When de-serializing, the default implementation of
     * {@link java.io.ObjectInputStream#resolveClass(ObjectStreamClass)} is used to resolve {@link Class} objects.
     *
     * @param clazz the {@link Class} of the instances being marshalled
     * @param <T>   the value type
     * @return a {@link Marshaller} for {@link Serializable} objects.
     */
    @Nonnull
    public static <T extends Serializable> Marshaller<T> serializableMarshaller(Class<T> clazz) {
        return new JavaSerializationMarshaller<>(clazz);
    }

    /**
     * Returns a {@link Marshaller} for {@link Serializable} objects. When de-serializing, the supplied
     * {@link ClassLoader} is used to resolve {@link Class} objects.
     *
     * @param clazz  the {@link Class} of the instances being marshalled
     * @param loader the {@link ClassLoader} is used to resolve {@link Class} objects
     * @param <T>    the value type
     * @return a {@link Marshaller} for {@link Serializable} objects.
     */
    @Nonnull
    public static <T extends Serializable> Marshaller<T> serializableMarshaller(Class<T> clazz, ClassLoader loader) {
        return new JavaSerializationMarshaller<>(clazz, loader);
    }
}
