/**
 * Represents an License object.
 */
UPM.define('LicenseModel', [
    'brace'
], function(Brace) {

    return Brace.Model.extend({
        namedAttributes: {
            active: 'boolean',
            autoRenewal: 'boolean',
            valid: 'boolean',
            evaluation: 'boolean',
            enterprise: 'boolean',   // Legacy
            maximumNumberOfUsers: 'number',
            licenseType: 'string',
            creationDateString: 'string',
            expiryDate: Date,
            expiryDateString: 'string',
            supportEntitlementNumber: 'string',
            organizationName: 'string',
            dataCenter: 'boolean',
            subscription: 'boolean',
            rawLicense: 'string',
            expired: 'boolean',
            maintenanceExpired: 'boolean',
            contactEmail: 'string'
        },

        /**
         * Calculate the time interval (rounded up to the next whole day) between the current datetime and
         * the license expiry datetime.
         * @param expiryDate A string or a Date object. Date and time in a day start-of boundary of 00:00:00.
         * @returns a decimal value. A negative day value means expired to the nearest day boundary.
         *
         * To avoid overwhelming numbers of comments. Only a few input and output examples below.
         * You can trace the following code with some console.log statements in an online JS tester.
         *
         * expiry: Apr 16 2018 00:00:00 GMT+1000 (AEST) local: 2018-04-15T07:00:00-07:00 (PST)
         * now:     2018-04-16T12:47:00-07:00 (PST) >  2018-04-15T07:00:00-07:00 (PST)
         * -1 day:  2018-04-15T12:47:00-07:00 (PST) >  2018-04-15T07:00:00-07:00 (PST)
         * -2 days: 2018-04-14T12:47:00-07:00 (PST) <  2018-04-15T07:00:00-07:00 (PST) STOP HERE
         * diff: -1787 minutes is -1 days  -347 minutes
         * adjust result to -2 days
         *
         * expiry: Apr 18 2018 00:00:00 GMT+1000 (AEST) local: 2018-04-17T07:00:00-07:00 (PST)
         * +1 day: 2018-04-17T12:47:00-07:00 (PST) > 2018-04-17T07:00:00-07:00 (PST) OVER
         * now:    2018-04-16T12:47:00-07:00 (PST) < 2018-04-17T07:00:00-07:00 (PST) STOP HERE
         * result:   0 days
         *
         * expiry: Apr 20 2018 00:00:00 GMT+1000 (AEST) local: 2018-04-19T07:00:00-07:00 (PST)
         * +3 days: 2018-04-19T12:47:00-07:00 (PST) > 2018-04-19T07:00:00-07:00 (PST) OVER
         * +2 days: 2018-04-18T12:47:00-07:00 (PST) < 2018-04-19T07:00:00-07:00 (PST) STOP HERE
         * +1 day:  2018-04-17T12:47:00-07:00 (PST) < 2018-04-19T07:00:00-07:00 (PST)
         * now:     2018-04-16T12:47:00-07:00 (PST) < 2018-04-19T07:00:00-07:00 (PST)
         * result:   2 days
         */
        daysUntilExpiry: function () {
            if (this.has('expiryDate')) {
                // The expiryDate is already adjusted to local date and time in a day start-of boundary of 00:00:00.
                // For example, when the license encoder license string maintenance expiration date is 2018-04-13:
                // expiryDate is printed as Fri Apr 13 2018 00:00:00 GMT+1000 (AEST) when the browser OS Date and Time Preference is Sydney.
                // expiryDate is printed as Thu Apr 12 2018 07:00:00 GMT-0700 (PDT) when the browser OS Date and Time Preference is San Francisco.
                const expiryDateString = this.get('expiryDate');
                const expiry = new Date(new Date(expiryDateString).setSeconds(0, 0));
                const now  = new Date(new Date().setSeconds(0, 0));
                const diffMinutes = Math.round((expiry - now) / 1000 / 60);
                const minutesPerDay = 60 * 24;
                return diffMinutes < 0 ?
                    Math.floor(diffMinutes / minutesPerDay): // largest day integer less than or equal to a negative diff.
                    Math.trunc(diffMinutes / minutesPerDay); // truncate to a positive day integer
            }
            return Infinity;
        },

        isExpiringThisWeek: function () {
            return !this.isExpired() && this.daysUntilExpiry() <= 6;
        },

        isExpiringTomorrow: function () {
            return !this.isExpired() && this.daysUntilExpiry() === 1;
        },

        isExpiringToday: function () {
            return !this.isExpired() && this.daysUntilExpiry() === 0;
        },

        isExpired: function () {
            return this.has('expired') ? this.getExpired() : this.getMaintenanceExpired();
        }
    });
});
