package com.atlassian.upgrade.spi;

import com.atlassian.annotations.PublicSpi;

import javax.annotation.Nonnull;
import java.util.Collection;

/**
 * SPI for a plugin or the host application to implement in order to be able to participate in upgrades.
 *
 * @since v1.0
 */
@PublicSpi
public interface UpgradeTaskFactory {

    /**
     * Returns the display name for this plugin / host app.
     * <p>e.g "JIRA Portfolio"
     * <p> This will be used to construct the user facing error message if the minimum build number check fails.
     *
     * @return The product's display name, i.e. the marketing name of the product to help user find
     * a way to upgrade to the minimal product version
     * @see #getProductMinimumVersion()
     */
    String getProductDisplayName();

    /**
     * Returns the minimum upgradeable version for this plugin / host app.
     * <p> e.g "7.0.0"
     * <p> This will be used to construct the user facing error message if the minimum build number check fails.
     *
     * @return The product's display name, i.e. the marketing name of the product to help user find
     * a way to upgrade to the minimal product version
     * @see #getProductMinimumVersion() 
     */
    String getProductMinimumVersion();

    /**
     * Get the minimum build number for a plugins data that can be upgraded from. There should be no upgrade tasks
     * with build numbers below this number.
     * <p>
     * For example, if the plugin data is at version 2 but the minimum build number is 4, it cannot be run. However,
     * if the plugin data is at version 6 than the upgrade can be completed.
     *
     * @return the minimum build number for this {@link UpgradeTaskFactory}
     */
    int getMinimumBuildNumber();

    /**
     * Get all of the upgrade tasks for this given plugin/host app. Each upgrade task should have a unique build number
     * and when executing the upgrade tasks they will be run in the numerical ordering.  If the build numbers are not
     * unique there is no guarantee of the ordering.
     *
     * @return upgrade tasks for this plugin/host app.
     */
    @Nonnull
    Collection<UpgradeTask> getAllUpgradeTasks();
}
