package com.atlassian.refapp.ctk.sal;

import com.atlassian.fugue.Option;
import com.atlassian.functest.junit.SpringAwareTestCase;
import com.atlassian.sal.api.usersettings.UserSettings;
import com.atlassian.sal.api.usersettings.UserSettingsBuilder;
import com.atlassian.sal.api.usersettings.UserSettingsService;
import com.google.common.base.Function;
import org.apache.commons.lang.StringUtils;
import org.junit.Before;
import org.junit.Test;

import javax.annotation.Nullable;

import static com.atlassian.sal.api.usersettings.UserSettingsService.MAX_KEY_LENGTH;
import static com.atlassian.sal.api.usersettings.UserSettingsService.MAX_STRING_VALUE_LENGTH;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

public class UserSettingsTest extends SpringAwareTestCase
{
    public static final String USERNAME = "admin";
    public static final String NOT_A_USERNAME = "notauser";
    public static final String KEY_1 = "key1";
    public static final String STRING_VALUE = "Hello";
    public static final String TOO_LONG_STRING = StringUtils.repeat("0", MAX_STRING_VALUE_LENGTH + 1);
    public static final String MAX_LENGTH_VALUE_STRING = TOO_LONG_STRING.substring(0, TOO_LONG_STRING.length() - 1);
    public static final String MAX_LENGTH_KEY_STRING = StringUtils.repeat("a", MAX_KEY_LENGTH);
    public static final long LONG_VALUE = 100l;
    private UserSettingsService userSettingsService;

    public void setService(UserSettingsService userSettingsService)
    {
        this.userSettingsService = userSettingsService;
    }

    @Before
    public void setUp()
    {
        // clear for user
        userSettingsService.updateUserSettings(USERNAME, new Function<UserSettingsBuilder, UserSettings>()
        {
            @Override
            public UserSettings apply(@Nullable UserSettingsBuilder input)
            {
                for (String key : input.getKeys())
                {
                    input.remove(key);
                }

                return input.build();
            }
        });
    }

    @Test
    public void userSettingsServiceShouldBeAvailable()
    {
        assertNotNull("UserSettingsService should be available to plugins", userSettingsService);
    }

    @Test(expected = IllegalArgumentException.class)
    public void noUserGetFailsWithCorrectException()
    {
        userSettingsService.getUserSettings(NOT_A_USERNAME);
    }

    @Test(expected = IllegalArgumentException.class)
    public void noUserUpdateFailsWithCorrectException()
    {
        userSettingsService.updateUserSettings(NOT_A_USERNAME, new Function<UserSettingsBuilder, UserSettings>()
        {
            @Override
            public UserSettings apply(@Nullable UserSettingsBuilder input)
            {
                // should not get here
                assertFalse(true);
                return input.build();
            }
        });
    }

    @Test
    public void typedBooleanUpdateStoresOnlyBooleanType()
    {
        // first insert a boolean
        userSettingsService.updateUserSettings(USERNAME, new Function<UserSettingsBuilder, UserSettings>()
        {
            @Override
            public UserSettings apply(UserSettingsBuilder input)
            {
                input.put(KEY_1, true);
                return input.build();
            }
        });

        // check that the stored value is there
        UserSettings userSettings = userSettingsService.getUserSettings(USERNAME);
        Option<Boolean> booleanValue = userSettings.getBoolean(KEY_1);
        assertFalse("boolean value should be present", booleanValue.isEmpty());
        assertTrue("boolean value should be true", booleanValue.get());

        // check the other value types are empty
        Option<String> stringValue = userSettings.getString(KEY_1);
        assertTrue("there should be no string type", stringValue.isEmpty());
        Option<Long> longValue = userSettings.getLong(KEY_1);
        assertTrue("there should be no long type", longValue.isEmpty());
    }

    @Test
    public void typedStringUpdateStoresOnlyStringType()
    {
        // now test a string overriding that boolean
        userSettingsService.updateUserSettings(USERNAME, new Function<UserSettingsBuilder, UserSettings>()
        {
            @Override
            public UserSettings apply(UserSettingsBuilder input)
            {
                input.put(KEY_1, STRING_VALUE);
                return input.build();
            }
        });

        // check that the stored value is there
        UserSettings userSettings = userSettingsService.getUserSettings(USERNAME);
        Option<String> stringValue = userSettings.getString(KEY_1);
        assertFalse(stringValue.isEmpty());
        assertEquals(STRING_VALUE, stringValue.get());

        // check the other value types are empty
        Option<Boolean> booleanValue = userSettings.getBoolean(KEY_1);
        assertTrue(booleanValue.isEmpty());
        Option<Long> longValue = userSettings.getLong(KEY_1);
        assertTrue(longValue.isEmpty());
    }

    @Test
    public void typedLongUpdateStoresOnlyLongType()
    {
        // now test a long
        userSettingsService.updateUserSettings(USERNAME, new Function<UserSettingsBuilder, UserSettings>()
        {
            @Override
            public UserSettings apply(UserSettingsBuilder input)
            {
                input.put(KEY_1, LONG_VALUE);
                return input.build();
            }
        });

        // check that the stored value is there
        UserSettings userSettings = userSettingsService.getUserSettings(USERNAME);
        Option<Long> longValue = userSettings.getLong(KEY_1);
        assertFalse(longValue.isEmpty());
        assertEquals(LONG_VALUE, (long) longValue.get());

        // check the other value types are empty
        Option<Boolean> booleanValue = userSettings.getBoolean(KEY_1);
        assertTrue(booleanValue.isEmpty());
        Option<String> stringValue = userSettings.getString(KEY_1);
        assertTrue(stringValue.isEmpty());

    }

    @Test
    public void updatingUserSettingsWithMaxLengthKeySucceeds()
    {
        userSettingsService.updateUserSettings(USERNAME, new Function<UserSettingsBuilder, UserSettings>()
        {
            @Override
            public UserSettings apply(UserSettingsBuilder input)
            {
                input.put(MAX_LENGTH_KEY_STRING, true);
                return input.build();
            }
        });
    }


    @Test(expected = IllegalArgumentException.class)
    public void updatingUserSettingsWithTooLongKeyFails()
    {
        userSettingsService.updateUserSettings(USERNAME, new Function<UserSettingsBuilder, UserSettings>()
        {
            @Override
            public UserSettings apply(UserSettingsBuilder input)
            {
                input.put(TOO_LONG_STRING, true);
                return input.build();
            }
        });
    }

    @Test
    public void updatingUserSettingsWithMaxLengthStringValueSucceeds()
    {
        userSettingsService.updateUserSettings(USERNAME, new Function<UserSettingsBuilder, UserSettings>()
        {
            @Override
            public UserSettings apply(UserSettingsBuilder input)
            {
                input.put(KEY_1, MAX_LENGTH_VALUE_STRING);
                return input.build();
            }
        });
    }

    @Test(expected = IllegalArgumentException.class)
    public void updatingUserSettingsWithTooLongStringValueFails()
    {
        userSettingsService.updateUserSettings(USERNAME, new Function<UserSettingsBuilder, UserSettings>()
        {
            @Override
            public UserSettings apply(UserSettingsBuilder input)
            {
                input.put(KEY_1, TOO_LONG_STRING);
                return input.build();
            }
        });
    }
}
