package com.atlassian.util.profiling;

import com.atlassian.util.profiling.strategy.MetricStrategy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;

import java.util.Collection;
import java.util.concurrent.TimeUnit;

import static java.util.Objects.requireNonNull;

/**
 * Factory and utility methods for create {@link MetricTimer} and {@link Histogram}.
 *
 * <pre>
 *     try (Ticker ignored = Metrics.startTimer("my-timer")) {
 *         // monitored code block here
 *     }
 * </pre>
 *
 * @since 3.0
 */
public class Metrics {
    private static final MetricsConfiguration CONFIGURATION = new MetricsConfiguration();

    private Metrics() {
        throw new UnsupportedOperationException("Metrics is an utility class and should not be instantiated");
    }

    @Nonnull
    public static MetricsConfiguration getConfiguration() {
        return CONFIGURATION;
    }

    @Nonnull
    public static Histogram histogram(@Nonnull String name) {
        return new DefaultHistogram(name);
    }

    @Nonnull
    public static MetricTimer timer(@Nonnull String name) {
        return new DefaultMetricTimer(requireNonNull(name, "name"));
    }

    @Nonnull
    public static Ticker startTimer(@Nonnull String name) {
        if (CONFIGURATION.isEnabled()) {
            return timer(name).start();
        }
        return Ticker.NO_OP;
    }

    private static class DefaultMetricTimer implements MetricTimer {
        private static final Logger log = LoggerFactory.getLogger(DefaultMetricTimer.class);

        private final String name;

        DefaultMetricTimer(String name) {
            this.name = name;
        }

        @Nonnull
        @Override
        public Ticker start() {
            Collection<MetricStrategy> metricStrategies = StrategiesRegistry.getMetricStrategies();
            if (!getConfiguration().isEnabled() || metricStrategies.isEmpty()) {
                return Ticker.NO_OP;
            }

            CompositeTicker compositeTicker = null;
            for (MetricStrategy strategy : metricStrategies) {
                try {
                    compositeTicker = Tickers.addTicker(strategy.startTimer(name), compositeTicker);
                } catch (Exception e) {
                    log.warn("Failed to start metric trace for {}", name, e);
                }
            }

            return compositeTicker == null ? Ticker.NO_OP : compositeTicker;
        }

        @Override
        public void update(long time, @Nonnull TimeUnit timeUnit) {
            if (getConfiguration().isEnabled()) {
                requireNonNull(timeUnit, "timeUnit");
                for (MetricStrategy strategy : StrategiesRegistry.getMetricStrategies()) {
                    try {
                        strategy.updateTimer(name, time, timeUnit);
                    } catch (Exception e) {
                        log.warn("Failed to update metric for {}", name, e);
                    }
                }
            }
        }
    }

    private static class DefaultHistogram implements Histogram {
        private static final Logger log = LoggerFactory.getLogger(DefaultHistogram.class);

        private final String name;

        private DefaultHistogram(String name) {
            this.name = name;
        }

        @Override
        public void update(long value) {
            if (getConfiguration().isEnabled()) {
                for (MetricStrategy strategy : StrategiesRegistry.getMetricStrategies()) {
                    try {
                        strategy.updateHistogram(name, value);
                    } catch (Exception e) {
                        log.warn("Failed to update histogram for {}", name, e);
                    }
                }
            }
        }
    }
}
