package com.atlassian.util.profiling.strategy.impl;

import com.atlassian.util.profiling.UtilTimerLogger;
import com.atlassian.util.profiling.UtilTimerStack;
import com.atlassian.util.profiling.strategy.ProfilingStrategy;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Default profiling strategy, which is included into {@link com.atlassian.util.profiling.UtilTimerStack} by default.
 * Before version 2.0 that was the only one available profiling functionality of the
 * {@link com.atlassian.util.profiling.UtilTimerStack}
 *
 * @since 2.0
 */
public class StackProfilingStrategy implements ProfilingStrategy
{
    private static final String ACTIVATE_MEMORY_PROPERTY = "atlassian.profile.activate.memory";
    private static final Logger log = LoggerFactory.getLogger(UtilTimerStack.class);
    /**
     * System property that controls the default maximum number of timer frames that can be reported per timer stack.
     */
    private static final String MAX_FRAME_COUNT = "atlassian.profile.maxframecount";
    /**
     * System property that controls the default threshold time below which a profiled event should not be reported
     */
    private static final String MIN_TIME = "atlassian.profile.mintime";
    /**
     * System property that controls the default threshold time below which an entire stack of profiled events should
     * not be reported.
     */
    private static final String MIN_TOTAL_TIME = "atlassian.profile.mintotaltime";
    /**
     * System property that specifies by default whether this timer should be used or not.  Set to "true" activates the
     * timer.  Set to "false" to deactivate.
     */
    private static final String ACTIVATE_PROPERTY = "atlassian.profile.activate";

    private static UtilTimerLogger logger = new UtilTimerLogger()
    {
        @Override
        public void log(String s)
        {
            log.debug(s);
        }
    };

    private final ThreadLocal<ProfilingTimerBean> current = new ThreadLocal<ProfilingTimerBean>();
    private boolean profileMemoryFlag;
    private long configuredMinTime;
    private long configuredMinTotalTime;
    private int configuredMaxFrameCount;
    private boolean enabled;

    public StackProfilingStrategy()
    {
        profileMemoryFlag = "true".equalsIgnoreCase(System.getProperty(ACTIVATE_MEMORY_PROPERTY, "false"));
        configuredMaxFrameCount = Integer.getInteger(MAX_FRAME_COUNT, 1500);
        configuredMinTime = Long.getLong(MIN_TIME, 0);
        configuredMinTotalTime = Long.getLong(MIN_TOTAL_TIME, 0);
        enabled = "true".equalsIgnoreCase(System.getProperty(ACTIVATE_PROPERTY, "false"));
    }

    @Override
    public void start(String name)
    {
        if (!isEnabled())
        {
            return;
        }

        //create a new timer and start it
        ProfilingTimerBean newTimer = new ProfilingTimerBean(name);
        newTimer.setStartTime();

        if (isProfileMemory())
        {
            newTimer.setStartMem();
        }

        //if there is a current timer - add the new timer as a child of it
        ProfilingTimerBean currentTimer = current.get();
        if (currentTimer != null)
        {
            currentTimer.addChild(newTimer);
            newTimer.setFrameCount(currentTimer.getFrameCount() + 1);
        }

        //set the new timer to be the current timer
        current.set(newTimer);
    }

    @Override
    public void stop(String name)
    {
        // We no longer check for isActive, as we want to cleanup the current stack of profiling beans
        ProfilingTimerBean currentTimer = current.get();
        if (currentTimer == null)
        {
            return;
        }

        currentTimer.setEndMem();

        //if the timers are matched up with each other (ie push("a"); pop("a"));
        if (name != null && name.equals(currentTimer.getResource()))
        {
            currentTimer.setEndTime();
            ProfilingTimerBean parent = currentTimer.getParent();
            //if we are the root timer, then print out the times
            if (parent == null)
            {
                if (currentTimer.getTotalTime() > getMinTotalTime())
                {
                    printTimes(currentTimer);
                }

                current.remove(); //for those servers that use thread pooling
            }
            else
            {
                if (currentTimer.getTotalTime() < getMinTime() || currentTimer.getFrameCount() > getMaxFrameCount())
                {
                    parent.removeChild(currentTimer);
                }

                parent.setFrameCount(currentTimer.getFrameCount());
                current.set(parent);
            }
        }
        else
        {
            //if timers are not matched up, then print what we have, and then print warning.
            printTimes(currentTimer);
            current.remove(); //prevent printing multiple times
            log.debug("Unmatched Timer.  Was expecting {}, instead got {}", currentTimer.getResource(), name);
        }
    }

    @Override
    public boolean isEnabled()
    {
        return enabled;
    }

    public void setEnabled(boolean enabled)
    {
        this.enabled = enabled;
    }

    public boolean isProfileMemory()
    {
        return profileMemoryFlag;
    }

    public long getMinTime()
    {
        return configuredMinTime;
    }

    public int getMaxFrameCount()
    {
        return configuredMaxFrameCount;
    }

    public long getMinTotalTime()
    {
        return configuredMinTotalTime;
    }

    public void setProfileMemoryFlag(boolean profileMemoryFlag)
    {
        this.profileMemoryFlag = profileMemoryFlag;
    }

    public void setConfiguredMinTime(long configuredMinTime)
    {
        this.configuredMinTime = configuredMinTime;
    }

    public void setConfiguredMaxFrameCount(int configuredMaxFrameCount)
    {
        this.configuredMaxFrameCount = configuredMaxFrameCount;
    }

    public void setConfiguredMinTotalTime(long configuredMinTotalTime)
    {
        this.configuredMinTotalTime = configuredMinTotalTime;
    }

    private void printTimes(ProfilingTimerBean currentTimer)
    {
        String printable = currentTimer.getPrintable(getMinTime());
        if (printable != null && !"".equals(printable.trim()))
        {
            logger.log(printable);
        }
    }

    public UtilTimerLogger getLogger()
    {
        return logger;
    }

    public void setLogger(UtilTimerLogger logger)
    {
        StackProfilingStrategy.logger = logger;
    }
}
