package com.atlassian.plugin.webresource.assembler;

import com.atlassian.plugin.webresource.impl.Globals;
import com.atlassian.plugin.webresource.impl.RequestState;
import com.atlassian.plugin.webresource.impl.UrlBuildingStrategy;
import com.atlassian.plugin.webresource.prebake.PrebakeWebResourceAssembler;
import com.atlassian.plugin.webresource.prebake.PrebakeWebResourceAssemblerBuilder;
import com.atlassian.plugin.webresource.util.TimeSpan;
import com.atlassian.webresource.api.assembler.WebResourceAssemblerBuilder;
import com.atlassian.webresource.api.prebake.Coordinate;

import java.util.Optional;
import java.util.concurrent.TimeUnit;

/**
 * Implementation of WebResourceAssemblerBuilder
 *
 * @since v3.0
 */
public class DefaultWebResourceAssemblerBuilder implements PrebakeWebResourceAssemblerBuilder {
    private final Globals globals;
    // Using optional to determine whether includeSuperbatchResources was invoked or not
    private Optional<Boolean> isSuperBatchingEnabled = Optional.empty();
    private Optional<Boolean> isSyncBatchingEnabled = Optional.empty();
    private Optional<Boolean> isAutoIncludeFrontendRuntimeEnabled = Optional.empty();
    private Optional<TimeSpan> deadline = Optional.empty();
    private Optional<Coordinate> coord = Optional.empty();

    public DefaultWebResourceAssemblerBuilder(final Globals globals) {
        this.globals = globals;
    }

    @Override
    public PrebakeWebResourceAssemblerBuilder withCoordinate(Coordinate coord) {
        this.coord = Optional.of(coord);
        return this;
    }

    @Override
    public PrebakeWebResourceAssembler build() {
        final UrlBuildingStrategy urlStrat = UrlBuildingStrategy.from(coord);
        final boolean autoIncludeFrontendRuntime = isAutoIncludeFrontendRuntimeEnabled.orElse(true);
        final RequestState requestState = new RequestState(globals, urlStrat, autoIncludeFrontendRuntime);

        final DefaultWebResourceAssembler assembler = new DefaultWebResourceAssembler(requestState, globals);

        deadline.ifPresent(timeSpan -> requestState.setBigPipeDeadline(System.currentTimeMillis() + timeSpan.toMillis()));

        isSyncBatchingEnabled.ifPresent(requestState::setSyncbatchEnabled);
        isSuperBatchingEnabled.ifPresent(requestState.getSuperbatchConfiguration()::setEnabled);

        return assembler;
    }

    /**
     * @deprecated since 5.5.0. Use {@link com.atlassian.webresource.api.assembler.RequiredResources#excludeSuperbatch()} instead.
     */
    @Deprecated
    @Override
    public WebResourceAssemblerBuilder includeSuperbatchResources(boolean include) {
        this.isSuperBatchingEnabled = Optional.of(include);
        return this;
    }

    @Override
    public WebResourceAssemblerBuilder includeSyncbatchResources(boolean include) {
        this.isSyncBatchingEnabled = Optional.of(include);
        return this;
    }

    @Override
    public WebResourceAssemblerBuilder autoIncludeFrontendRuntime(boolean include) {
        this.isAutoIncludeFrontendRuntimeEnabled = Optional.of(include);
        return this;
    }

    @Override
    public WebResourceAssemblerBuilder asyncDataDeadline(long deadline, TimeUnit timeunit) {
        this.deadline = Optional.of(new TimeSpan(deadline, timeunit));
        return this;
    }
}
