package com.atlassian.plugin.webresource.assembler;

import com.atlassian.plugin.webresource.impl.UrlBuildingStrategy;
import com.atlassian.plugin.webresource.impl.config.Config;
import com.atlassian.plugin.webresource.impl.Globals;
import com.atlassian.plugin.webresource.impl.RequestState;
import com.atlassian.plugin.webresource.prebake.PrebakeWebResourceAssembler;
import com.atlassian.plugin.webresource.prebake.PrebakeWebResourceAssemblerBuilder;
import com.atlassian.plugin.webresource.util.TimeSpan;
import com.atlassian.webresource.api.assembler.WebResourceAssemblerBuilder;
import com.atlassian.webresource.api.prebake.Coordinate;

import java.util.Optional;
import java.util.concurrent.TimeUnit;

/**
 * Implementation of WebResourceAssemblerBuilder
 * @since v3.0
 */
public class DefaultWebResourceAssemblerBuilder implements PrebakeWebResourceAssemblerBuilder
{
    private final Globals globals;
    private boolean isSuperBatchingEnabled;
    private Optional<TimeSpan> deadline = Optional.empty();
    private Optional<Coordinate> coord = Optional.empty();

    public DefaultWebResourceAssemblerBuilder(Globals globals)
    {
        this.globals = globals;
        this.isSuperBatchingEnabled = globals.getConfig().isSuperBatchingEnabled();
    }

    @Override
    public PrebakeWebResourceAssemblerBuilder withCoordinate(Coordinate coord)
    {
        this.coord = Optional.of(coord);
        return this;
    }

    @Override
    public PrebakeWebResourceAssembler build()
    {
        UrlBuildingStrategy urlStrat = UrlBuildingStrategy.from(coord);
        RequestState requestState = new RequestState(globals, urlStrat);
        if (deadline.isPresent()) {
            requestState.setBigPipeDeadline(System.currentTimeMillis() + deadline.get().toMillis());
        }
        DefaultWebResourceAssembler assembler = new DefaultWebResourceAssembler(requestState, globals.getConfig());
        if (isSuperBatchingEnabled)
        {
            assembler.resources().requireContext(Config.SUPER_BATCH_CONTEXT_KEY);
        }
        return assembler;
    }

    @Override
    public WebResourceAssemblerBuilder includeSuperbatchResources(boolean include)
    {
        this.isSuperBatchingEnabled = include;
        return this;
    }

    @Override
    public WebResourceAssemblerBuilder asyncDataDeadline(long deadline, TimeUnit timeunit) {
        this.deadline = Optional.of(new TimeSpan(deadline, timeunit));
        return this;
    }
}
