package com.atlassian.plugin.webresource;

import com.atlassian.plugin.webresource.impl.config.Config;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringEscapeUtils;
import org.apache.commons.lang.StringUtils;

import java.util.Arrays;
import java.util.List;
import java.util.Map;

/**
 * Outputs all URLs as {@code <link rel="prefetch" href="theurl">}.
 *
 * @see com.atlassian.webresource.api.assembler.WebResourceSet#writePrefetchLinks
 * @since 3.6.0
 */
public class PrefetchLinkWebResource extends AbstractWebResourceFormatter {
    public static final WebResourceFormatter FORMATTER = new PrefetchLinkWebResource();

    // crossorigin is supported for prefetch in Firefox: https://developer.mozilla.org/en-US/docs/Web/HTML/Link_types
    private static final List<String> HANDLED_PARAMETERS = Arrays.asList("title", Config.MEDIA_PARAM_NAME, "crossorigin");

    public boolean matches(String name) {
        return name != null && (
                name.endsWith(Config.JS_EXTENSION) || name.endsWith(Config.CSS_EXTENSION) ||
                        name.endsWith(Config.LESS_EXTENSION) || name.endsWith(Config.SOY_EXTENSION)
        );
    }

    public String formatResource(String url, Map<String, String> attributes) {
        if (attributes.containsKey(Config.CONDITIONAL_COMMENT_PARAM_NAME)
                || BooleanUtils.toBoolean(attributes.get(Config.IEONLY_PARAM_NAME))) {
            // Refuse to output a prefetch link if this is resource is normally served with IE conditional comments.
            return "";
        }

        StringBuilder buffer = new StringBuilder("<link rel=\"prefetch\" href=\"");
        buffer.append(StringEscapeUtils.escapeHtml(url)).append("\"");

        List attributeTokens = getParametersAsAttributes(attributes);
        if (attributes.size() > 0) {
            buffer.append(" ").append(StringUtils.join(attributeTokens.iterator(), " "));
        }

        buffer.append(">\n");

        return buffer.toString();
    }

    protected List<String> getAttributeParameters() {
        return HANDLED_PARAMETERS;
    }
}
