package com.atlassian.plugin.webresource;

import com.atlassian.plugin.webresource.url.UrlParameters;

import java.util.Map;
import java.util.Set;

/**
 * Represents a plugin resource.
 * This object represents a page-time resource. It is used build a URL to the underlying resource,
 * but does not know how to get the contents of that resource itself.
 *
 * @since 2.2
 */
public interface PluginResource
{
    /**
     * @return true if caching for this resource is supported. Use this check to append a static
     * caching url prefix to this resource's url.
     */
    boolean isCacheSupported();

    /**
     * @return the url for this plugin resource, never null.
     */
    String getUrl();

    /**
     * @return the resource name for the plugin resource.
     */
    String getResourceName();

    /**
     * @return the plugin module's complete key for which this resource belongs to.
     */
    String getCompleteKey();

    /**
     * @return url parameters
     */
    UrlParameters getUrlParameters();

    /**
     * @return a map of parameter key and value pairs for this resource, can be empty but never null. Resources that
     * have any values in this map will be included in a separate batch.
     */
    Map<String, String> getParams();

    /**
     * @return a map of parameter key and value pairs for this resource merged with batchable params, can be empty
     * but never null. Including a resource in this map will not cause the resource to be excluded from a batch.
     */
    Map<String, String> getParamsWithBatchableParams();

    /**
     * @return the version prefix string for a cached resource
     */
    String getVersion();

    /**
     * @return the type (e.g. file extension) of the resource.
     * @since 2.12.3
     */
    String getType();
    
    /**
     * This method provides the mechanism for discovering the contents of a PluginResource.
     * <p/>
     * The primary use of this method is in providing a representation of the contents 
     * (for dependency tracking) when formatting a PluginResource.
     * 
     * @return a Set of descriptors providing information about the contents of this PluginResource.
     */
    Set<BatchedWebResourceDescriptor> getBatchedWebResourceDescriptors();

    /**
     * @param isAbsolute should generated url be absolute or relative.
     * @return the url for this plugin resource, never null.
     */
    String getUrl(boolean isAbsolute);

    /**
     * @return hash of all contributed hashes.
     */
    String getHash();

    /**
     * Describes whether a resource can be safely delivered via a stateless route like a CDN.
     * Such routes will drop cookies (CDNs do this, as to cookie-less domains), etc. This means
     * that any state the resource is relying on (e.g. current user) will not be preserved
     * between page-generation-time and request-delivery-time.
     *
     * Typically, a resource is safe for CDN delivery if non of the transforms are legacy transforms,
     * and none of the conditions are legacy conditions. This means using UrlReadingWebResourceTransformer and
     * UrlReadingCondition, which capture their state in the url and are hence otherwise stateless.
     *
     * @since 3.0.5
     */
    boolean isCdnSupported();
}