package com.atlassian.plugin.webresource;

import com.atlassian.plugin.ModuleDescriptor;
import com.atlassian.plugin.util.PluginUtils;
import com.atlassian.plugin.webresource.condition.ConditionsCache;
import com.atlassian.util.concurrent.ResettableLazyReference;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;

import com.atlassian.plugin.webresource.condition.ConditionState;
import com.atlassian.plugin.webresource.condition.DecoratingCondition;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Iterables.contains;
import static com.google.common.collect.Iterables.filter;
import static java.util.Collections.emptyMap;
import static java.util.Collections.unmodifiableCollection;
import static java.util.Collections.unmodifiableMap;

public class DefaultResourceDependencyResolver implements ResourceDependencyResolver
{
    public static final String IMPLICIT_CONTEXT_NAME = "_super".toString();

    private static final Logger log = LoggerFactory.getLogger(DefaultResourceDependencyResolver.class);

    private final WebResourceIntegration webResourceIntegration;
    private final ResourceBatchingConfiguration batchingConfiguration;
    private final Cache cached = new Cache();

    public DefaultResourceDependencyResolver(final WebResourceIntegration webResourceIntegration, final ResourceBatchingConfiguration batchingConfiguration)
    {
        this.webResourceIntegration = webResourceIntegration;
        this.batchingConfiguration = batchingConfiguration;
    }

    public Iterable<WebResourceModuleDescriptor> getSuperBatchDependencies(ConditionState conditionsRun)
    {
        SuperBatch superBatch = cached.lazy.get();
        conditionsRun.addAll(superBatch.conditionsRun);
        return cached.resourceMap().values();
    }

    private Iterable<String> getSuperBatchDependencyKeys()
    {
        return cached.resourceMap().keySet();
    }

    @Override
    public Iterable<WebResourceModuleDescriptor> getDependencies(final String moduleKey,
                                                                 final boolean excludeSuperBatchedResources,
                                                                 ConditionState conditionsRun,
                                                                 ConditionsCache conditionsCache)
    {
        final LinkedHashMap<String, WebResourceModuleDescriptor> orderedResources = new LinkedHashMap<String, WebResourceModuleDescriptor>();
        final Iterable<String> superBatchResources = excludeSuperBatchedResources ? getSuperBatchDependencyKeys() : Collections.<String> emptyList();
        resolveDependencies(moduleKey, orderedResources, superBatchResources, new Stack<String>(), null, conditionsRun, conditionsCache);
        return orderedResources.values();
    }

    @Override
    public Iterable<WebResourceModuleDescriptor> getDependenciesInContext(final String context,
                                                                          ConditionState conditionsRun,
                                                                          ConditionsCache conditionsCache)
    {
        return getDependenciesInContext(context, new LinkedHashSet<String>(), conditionsRun, conditionsCache);
    }

    @Override
    public Iterable<WebResourceModuleDescriptor> getDependenciesInContext(final String context, boolean excludeSuperBatchedResources,
                                                                          ConditionState conditionsRun,
                                                                          ConditionsCache conditionsCache)
    {
        return getDependenciesInContext(context, excludeSuperBatchedResources, new LinkedHashSet<String>(), conditionsRun, conditionsCache);
    }

    @Override
    public Iterable<WebResourceModuleDescriptor> getDependenciesInContext(final String context,
                                                                          final Set<String> skippedResources,
                                                                          ConditionState conditionsRun,
                                                                          ConditionsCache conditionsCache)
    {
        return getDependenciesInContext(context, true, skippedResources, conditionsRun, conditionsCache);
    }

    @Override
    public Iterable<WebResourceModuleDescriptor> getDependenciesInContext(final String context,
                                                                          boolean excludeSuperBatchedResources,
                                                                          final Set<String> skippedResources,
                                                                          ConditionState conditionsRun,
                                                                          ConditionsCache conditionsCache)
    {
        final Set<WebResourceModuleDescriptor> contextResources = new LinkedHashSet<WebResourceModuleDescriptor>();
        for (final WebResourceModuleDescriptor descriptor : modulesInContext(context, conditionsRun))
        {
            final LinkedHashMap<String, WebResourceModuleDescriptor> dependencies = new LinkedHashMap<String, WebResourceModuleDescriptor>();
            final Iterable<String> superBatchResources = excludeSuperBatchedResources ? getSuperBatchDependencyKeys() : Collections.<String> emptyList();
            resolveDependencies(descriptor.getCompleteKey(), dependencies, superBatchResources, new Stack<String>(), skippedResources, conditionsRun, conditionsCache);
            for (final WebResourceModuleDescriptor dependency : dependencies.values())
            {
                contextResources.add(dependency);
            }
        }
        return unmodifiableCollection(contextResources);
    }

    private Iterable<? extends WebResourceModuleDescriptor> modulesInContext(final String context, ConditionState conditionsRun)
    {
        if (IMPLICIT_CONTEXT_NAME.equals(context))
        {
            return getSuperBatchDependencies(conditionsRun);
        }
        else
        {
            final Class<WebResourceModuleDescriptor> clazz = WebResourceModuleDescriptor.class;
            return filter(webResourceIntegration.getPluginAccessor().getEnabledModuleDescriptorsByClass(clazz), new Predicate<WebResourceModuleDescriptor>()
            {
                @Override
                public boolean apply(@Nullable WebResourceModuleDescriptor descriptor)
                {
                    return descriptor.getContexts().contains(context);
                }
            });
        }
    }

    /**
     * Adds the resources as well as its dependencies in order to the given ordered set. This method uses recursion
     * to add a resouce's dependent resources also to the set. You should call this method with a new stack
     * passed to the last parameter.
     *
     * Note that resources already in the given super batch will be excluded when resolving dependencies. You
     * should pass in an empty set for the super batch to include super batch resources.
     * @param moduleKey the module complete key to add as well as its dependencies
     * @param orderedResourceKeys an ordered list set where the resources are added in order
     * @param superBatchResources the set of super batch resources to exclude when resolving dependencies
     * @param stack where we are in the dependency tree
     * @param skippedResources if not null, all resources with conditions are skipped and added to this set.
     * @param conditionsRun
     */
    private void resolveDependencies(final String moduleKey,
                                     final Map<String, WebResourceModuleDescriptor> orderedResourceKeys,
                                     final Iterable<String> superBatchResources, final Stack<String> stack,
                                     final Set<String> skippedResources, ConditionState conditionsRun, ConditionsCache conditionsCache)
    {
        if (contains(superBatchResources, moduleKey))
        {
            log.debug("Not requiring resource: {} because it is part of a super-batch", moduleKey);
            return;
        }
        if (stack.contains(moduleKey))
        {
            log.warn("Cyclic plugin resource dependency has been detected with: {} \nStack trace: {}", moduleKey, stack);
            return;
        }

        final ModuleDescriptor<?> moduleDescriptor;
        try
        {
            moduleDescriptor = webResourceIntegration.getPluginAccessor().getEnabledPluginModule(moduleKey);
        }
        catch (IllegalArgumentException e)
        {
            // this can happen if the module key is an invalid format (e.g. has no ":")
            log.warn("Cannot find web resource module for: {}", moduleKey);
            return;
        }
        if (!(moduleDescriptor instanceof WebResourceModuleDescriptor))
        {
            if (Boolean.getBoolean(PluginUtils.ATLASSIAN_DEV_MODE))
            {
                if (webResourceIntegration.getPluginAccessor().getPluginModule(moduleKey) != null)
                {
                    log.warn("Cannot include disabled web resource module: {}", moduleKey);
                }
                else
                {
                    log.warn("Cannot find web resource module for: {}", moduleKey);
                }
            }
            return;
        }

        final WebResourceModuleDescriptor webResourceModuleDescriptor = (WebResourceModuleDescriptor) moduleDescriptor;

        // If the webresource cannot encode its state into the URL, it must be broken out of the batch and served
        // separately. This is normally due to the presence of a DecoratingLegacyCondition
        boolean skipDependencies = false;
        if (!webResourceModuleDescriptor.canEncodeStateIntoUrl())
        {
            if (null != skippedResources)
            {
                skippedResources.add(moduleKey);
                return;
            }
            else
            {
                DecoratingCondition condition = webResourceModuleDescriptor.getCondition();
                boolean displayImmediate = webResourceModuleDescriptor.shouldDisplayImmediate(conditionsCache);
                if (condition != null)
                {
                    conditionsRun.addCondition(condition, displayImmediate);
                }
                if (!displayImmediate)
                {
                    // don't include this web resource, or any of its dependencies
                    log.debug("Cannot include web resource module {} as its condition fails", moduleDescriptor.getCompleteKey());
                    return;
                }
                conditionsRun.addWebResourceModuleDescriptor(webResourceModuleDescriptor);
            }
        }
        else
        {
            DecoratingCondition condition = webResourceModuleDescriptor.getCondition();
            if (condition != null)
            {
                boolean displayImmediate = webResourceModuleDescriptor.shouldDisplayImmediate(conditionsCache);
                conditionsRun.addCondition(condition, displayImmediate);
                conditionsRun.addWebResourceModuleDescriptor(webResourceModuleDescriptor);
                skipDependencies = !displayImmediate;
            }
        }

        final List<String> dependencies = webResourceModuleDescriptor.getDependencies();
        log.debug("About to add resource [{}] and its dependencies: {}", moduleKey, dependencies);
        stack.push(moduleKey);
        try
        {
            if (!skipDependencies)
            {
                for (final String dependency : dependencies)
                {
                    if (orderedResourceKeys.get(dependency) == null)
                    {
                        resolveDependencies(dependency, orderedResourceKeys, superBatchResources, stack, skippedResources,
                            conditionsRun, conditionsCache);
                    }
                }
            }
        }
        finally
        {
            stack.pop();
        }
        orderedResourceKeys.put(moduleKey, webResourceModuleDescriptor);
    }

    final class Cache
    {
        ResettableLazyReference<SuperBatch> lazy = new ResettableLazyReference<SuperBatch>()
        {
            @Override
            protected SuperBatch create() throws Exception
            {
                // The linked hash map ensures that order is preserved
                final String version = webResourceIntegration.getSuperBatchVersion();
                ConditionState conditionsRun = new ConditionState();
                ConditionsCache conditionsCache = new ConditionsCache();
                Map<String, WebResourceModuleDescriptor> resources = loadDescriptors(
                    batchingConfiguration.getSuperBatchModuleCompleteKeys(), conditionsRun, conditionsCache);
                return new SuperBatch(version, resources, conditionsRun);
            }

            Map<String, WebResourceModuleDescriptor> loadDescriptors(final Iterable<String> moduleKeys,
                                                                     ConditionState conditionsRun, ConditionsCache conditionsCache)
            {
                if (Iterables.isEmpty(moduleKeys))
                {
                    return emptyMap();
                }
                final Map<String, WebResourceModuleDescriptor> resources = new LinkedHashMap<String, WebResourceModuleDescriptor>();
                for (final String moduleKey : moduleKeys)
                {
                    resolveDependencies(moduleKey, resources, Collections.<String> emptyList(), new Stack<String>(),
                        null, conditionsRun, conditionsCache);
                }
                return unmodifiableMap(resources);
            }
        };

        Map<String, WebResourceModuleDescriptor> resourceMap()
        {
            if (!batchingConfiguration.isSuperBatchingEnabled())
            {
                log.debug("Super batching not enabled, but getSuperBatchDependencies() called. Returning empty.");
                return emptyMap();
            }
            while (true)
            {
                final SuperBatch batch = lazy.get();
                if (batch.version.equals(webResourceIntegration.getSuperBatchVersion()))
                {
                    return batch.resources;
                }

                // The super batch has been updated so recreate the batch
                lazy.reset();
            }
        }
    }

    static final class SuperBatch
    {
        final String version;
        final Map<String, WebResourceModuleDescriptor> resources;
        final ConditionState conditionsRun;

        SuperBatch(final String version, final Map<String, WebResourceModuleDescriptor> resources,
                   ConditionState conditionsRun)
        {
            this.conditionsRun = conditionsRun;
            this.version = checkNotNull(version);
            this.resources = checkNotNull(resources);
        }
    }
}
