package com.atlassian.plugin.webresource;

import com.atlassian.annotations.Internal;
import com.atlassian.plugin.ModuleDescriptor;
import com.atlassian.plugin.Plugin;
import com.atlassian.plugin.elements.ResourceDescriptor;
import com.atlassian.plugin.elements.ResourceLocation;
import com.atlassian.plugin.servlet.AbstractFileServerServlet;
import com.atlassian.plugin.servlet.ContentTypeResolver;
import com.atlassian.plugin.servlet.DownloadException;
import com.atlassian.plugin.servlet.DownloadableClasspathResource;
import com.atlassian.plugin.servlet.DownloadableResource;
import com.atlassian.plugin.servlet.DownloadableWebResource;
import com.atlassian.plugin.servlet.ServletContextFactory;
import com.atlassian.plugin.util.PluginUtils;
import com.atlassian.plugin.util.validation.ValidationException;
import com.atlassian.plugin.webresource.condition.DecoratingCondition;
import com.atlassian.plugin.webresource.transformer.StaticTransformers;
import com.atlassian.plugin.webresource.transformer.TransformerCache;
import com.atlassian.plugin.webresource.util.HashBuilder;
import com.atlassian.sourcemap.SourceMap;
import com.atlassian.sourcemap.Util;
import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableList;
import org.apache.commons.lang.StringUtils;

import java.io.File;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.atlassian.plugin.webresource.support.http.BaseRouter.joinWithSlashWithoutEmpty;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * WARNING Do not use it, it will be removed in the next version!
 *
 * Unites all the configurations in one place. Not exposed as API, so it is possible to change it and store settings
 * that should not be exposed publicly.
 *
 * @since 3.3
 */
@Internal
public class Config
{
    public static final String RESOURCE_SOURCE_PARAM = "source";
    public static final String CONTEXT_RESOURCE_PREFIX = "_context";
    public static final String RESOURCE_BATCH_PARAM = "batch";

    private static final String RESOURCE_DOWNLOAD_TYPE = "download";

    private final WebResourceIntegration integration;
    private final ResourceBatchingConfiguration batchingConfiguration;
    private final WebResourceUrlProvider urlProvider;
    private ContentTypeResolver contentTypeResolver;
    private final ServletContextFactory servletContextFactory;
    private final StaticTransformers staticTransformers;
    private final TransformerCache transformerCache;

    public Config(ResourceBatchingConfiguration batchingConfiguration, WebResourceIntegration integration,
            WebResourceUrlProvider urlProvider, ServletContextFactory servletContextFactory,
            StaticTransformers staticTransformers, TransformerCache transformerCache)
    {
        this.batchingConfiguration = batchingConfiguration;
        this.integration = integration;
        this.urlProvider = urlProvider;
        this.servletContextFactory = servletContextFactory;
        this.staticTransformers = staticTransformers;
        this.transformerCache = transformerCache;
    }

    /**
     * Needed because `contentTypeResolver` isn't available at the time Config is created and set up a bit later.
     */
    public void setContentTypeResolver(ContentTypeResolver contentTypeResolver)
    {
        if (this.contentTypeResolver != null)
        {
            throw new RuntimeException("content type resolver already set!");
        }
        this.contentTypeResolver = contentTypeResolver;
    }

    /**
     * @return if cache should be enabled.
     */
    public boolean isCacheEnabled()
    {
        return !Boolean.getBoolean(PluginUtils.WEBRESOURCE_DISABLE_FILE_CACHE) || Boolean.getBoolean(PluginUtils
                .ATLASSIAN_DEV_MODE);
    }

    /**
     * @return amount of files to be stored in cache.
     */
    public int getCacheSize()
    {
        return Integer.getInteger(PluginUtils.WEBRESOURCE_FILE_CACHE_SIZE, 1000);
    }

    /**
     * @return directory where cache content would be stored as files.
     */
    public File getCacheDirectory()
    {
        return integration.getTemporaryDirectory();
    }

    /**
     * @param typeOrContentType type or content type.
     * @return if source map enabled for this type or content type.
     */
    public boolean isSourceMapEnabledFor(String typeOrContentType)
    {
        return batchingConfiguration.isSourceMapEnabled() && Util.isSourceMapSupportedBy(typeOrContentType);
    }

    /**
     * @return if CDN enabled.
     */
    public boolean isCdnEnabled()
    {
        return null != integration.getCDNStrategy() && integration.getCDNStrategy().supportsCdn();
    }

    /**
     * @return get current locale.
     */
    public String getCurrentLocale()
    {
        return integration.getStaticResourceLocale();
    }

    /**
     * @return base url, where web resources is mounted to.
     */
    public String getBaseUrl()
    {
        return getBaseUrl(true);
    }

    /**
     * @param isAbsolute if url should be absolute or relative.
     * @return base url, where web resources is mounted to.
     */
    public String getBaseUrl(boolean isAbsolute)
    {
        try {
            return joinWithSlashWithoutEmpty(urlProvider.getBaseUrl(isAbsolute ? UrlMode.ABSOLUTE : UrlMode.RELATIVE),
                AbstractFileServerServlet.SERVLET_PATH);
        }
        catch (AssertionError e)
        {
            // For unknown reason some links in Confluence (like /confluence/setup/setupstart.action ) doesn't
            // support the absolute base url, catching such cases and trying to use relative base url.
            if (isAbsolute && e.getMessage().contains("Unsupported URLMode"))
            {
                return getBaseUrl(false);
            }
            else
            {
                throw e;
            }
        }
    }

    /**
     * @return url prefix for resource, including base url.
     */
    public String getResourceUrlPrefix(String hash, String version, boolean isAbsolute)
    {
        return urlProvider.getStaticResourcePrefix(hash, version, (isAbsolute ? UrlMode.ABSOLUTE : UrlMode.RELATIVE))
                + "/" + AbstractFileServerServlet.SERVLET_PATH;
    }

    /**
     * @return url prefixed with CDN prefix.
     */
    public String getResourceCdnPrefix(String url)
    {
        return integration.getCDNStrategy().transformRelativeUrl(url);
    }

    /**
     * @return content type for given path.
     */
    public String getContentType(String path)
    {
        return contentTypeResolver.getContentType(path);
    }

    /**
     * @return content annotators for given type.
     */
    public ResourceContentAnnotator[] getContentAnnotators(String type)
    {
        if ("js".equals(type))
        {
            if (batchingConfiguration.isJavaScriptTryCatchWrappingEnabled())
            {
                return new ResourceContentAnnotator[] { new SemicolonResourceContentAnnotator(),
                        new TryCatchJsResourceContentAnnotator(), new LocationContentAnnotator() };
            }
            else
            {
                return new ResourceContentAnnotator[] { new SemicolonResourceContentAnnotator(),
                        new LocationContentAnnotator() };
            }
        }
        else if ("css".equals(type))
        {
            return new ResourceContentAnnotator[] { new LocationContentAnnotator() };
        }
        else
        {
            return new ResourceContentAnnotator[] { };
        }
    }

    /**
     * @return static transformers.
     */
    public StaticTransformers getStaticTransformers()
    {
        return staticTransformers;
    }

    /**
     * Internal data structure used to cache list of web resources and its transformers and conditions.
     */
    protected static class Snapshot
    {
        final Config config;

        Snapshot(Config config)
        {
            this.config = config;
        }

        // Storing some attribute of Resource outside of it in a separate Set in order to consume less memory
        // because amount of such attributes should be much smaller than the amount of web resources.
        final Map<Bundle, List<WebResourceTransformation>> webResourcesTransformations = new HashMap<Bundle,
                List<WebResourceTransformation>>();
        final Map<Bundle, DecoratingCondition> webResourcesCondition = new HashMap<Bundle, DecoratingCondition>();
        final Set<Bundle> webResourcesWithLegacyConditions = new HashSet<Bundle>();
        final Set<Bundle> webResourcesWithLegacyTransformers = new HashSet<Bundle>();
    }

    /**
     * Analyzes current state of the plugin system and return web resources.
     */
    protected void ensureNoLegacyStuff(Snapshot snapshot)
    {
        if (integration.forbidCondition1AndTransformer1())
        {
            // Checking for legacy conditions.
            List<String> resourcesWithLegacyConditions = new LinkedList<String>();
            for (Bundle bundle : snapshot.webResourcesWithLegacyConditions)
            {
                if (!integration.allowedCondition1Keys().contains(bundle.getKey()))
                {
                    resourcesWithLegacyConditions.add(bundle.getKey());
                }
            }

            // Checking for legacy transformers.
            List<String> resourcesWithLegacyTransformers = new LinkedList<String>();
            for (Bundle bundle : snapshot.webResourcesWithLegacyTransformers)
            {
                if (!integration.allowedTransform1Keys().contains(bundle.getKey()))
                {
                    resourcesWithLegacyTransformers.add(bundle.getKey());
                }
            }

            if (resourcesWithLegacyConditions.size() > 0 || resourcesWithLegacyTransformers.size() > 0)
            {
                List<String> messages = new ArrayList<String>();
                if (resourcesWithLegacyConditions.size() > 0)
                {
                    messages.add("legacy conditions: \"" + Joiner.on("\", \"").join(resourcesWithLegacyConditions) + "\"");
                }
                if (resourcesWithLegacyTransformers.size() > 0)
                {
                    messages.add("legacy transformers: \"" + Joiner.on("\", \"").join(resourcesWithLegacyTransformers) + "\"");
                }
                throw new ValidationException("there are web resources with " + Joiner.on(", and ").join(messages)
                    , ImmutableList.<String>of());
            }
        }
    }

    /**
     * Analyzes current state of the plugin system and return web resources.
     */
    public Map<String, Bundle> getWebResourcesWithoutCache()
    {
        class IntermediaryContextData
        {
            public List<String> dependencies = new ArrayList<String>();
            public Date updatedAt;
            public String version = "";
        }

        Snapshot snapshot = new Snapshot(this);
        Map<String, Bundle> bundles = new HashMap<String, Bundle>();

        // Collecting web resource descriptors.
        Map<String, IntermediaryContextData> intermediaryContexts = new HashMap<String, IntermediaryContextData>();
        List<WebResourceModuleDescriptor> webResourceDescriptors = integration.getPluginAccessor()
                .getEnabledModuleDescriptorsByClass(WebResourceModuleDescriptor.class);

        // Processing descriptors and building web resources graph.
        for (final WebResourceModuleDescriptor webResourceDescriptor : webResourceDescriptors)
        {
            Plugin plugin = webResourceDescriptor.getPlugin();
            Date updatedAt = (plugin.getDateLoaded() == null) ? new Date() : plugin.getDateLoaded();

            // Checking for legacy conditions.
            String completeKey = webResourceDescriptor.getCompleteKey();
            DecoratingCondition condition = webResourceDescriptor.getCondition();
            boolean hasLegacyConditions = (condition != null) && !condition.canEncodeStateIntoUrl();

            // Checking for legacy transformers.
            List<WebResourceTransformation> transformations = webResourceDescriptor.getTransformations();
            boolean hasLegacyTransformers = false;
            // Strictly speaking we should check the extension of the transformer,
            // because legacy CSS transformers would not affect JS.
            // But, it would hurt the performance to check transformers against all the resources in the web resource
            // . So, to make it
            // faster we just check if there's any legacy transformer at all.
            for (WebResourceTransformation transformation : transformations)
            {
                if (!transformation.containsOnlyPureUrlReadingTransformers(transformerCache))
                {
                    hasLegacyTransformers = true;
                    break;
                }
            }

            // Adding web resource.
            Bundle bundle = new WebResource(snapshot, completeKey, webResourceDescriptor.getDependencies(),
                    updatedAt, plugin.getPluginInformation().getVersion(), true);
            if (hasLegacyConditions)
            {
                snapshot.webResourcesWithLegacyConditions.add(bundle);
            }
            if (hasLegacyTransformers)
            {
                snapshot.webResourcesWithLegacyTransformers.add(bundle);
            }
            bundles.put(completeKey, bundle);

            // Collecting dependencies and dates for contexts, we would need this information later when transform
            // contexts into
            // virtual web resources.
            for (String context : webResourceDescriptor.getContexts())
            {
                if (!context.equals(completeKey))
                {
                    String contextResourceKey = CONTEXT_RESOURCE_PREFIX + ":" + context;
                    IntermediaryContextData contextData = intermediaryContexts.get(contextResourceKey);
                    if (contextData == null)
                    {
                        contextData = new IntermediaryContextData();
                        intermediaryContexts.put(contextResourceKey, contextData);
                    }
                    contextData.dependencies.add(completeKey);
                    if (contextData.updatedAt == null || contextData.updatedAt.before(updatedAt))
                    {
                        contextData.updatedAt = updatedAt;
                    }
                    contextData.version = HashBuilder.buildHash(contextData.version, bundle.getVersion());
                }
            }

            // Adding transformations and conditions.
            snapshot.webResourcesCondition.put(bundle, condition);
            snapshot.webResourcesTransformations.put(bundle, transformations);
        }

        // Turning contexts into virtual resources.
        for (Map.Entry<String, IntermediaryContextData> entry : intermediaryContexts.entrySet())
        {
            String contextResourceKey = entry.getKey();
            IntermediaryContextData contextData = entry.getValue();
            bundles.put(contextResourceKey, new Bundle(snapshot, contextResourceKey, contextData.dependencies,
                    contextData.updatedAt, contextData.version, true));
        }

        // Adding super batch as virtual resource.
        if (batchingConfiguration.isSuperBatchingEnabled())
        {
            List<String> dependencies = batchingConfiguration.getSuperBatchModuleCompleteKeys();
            // If super batch is empty setting very old date as it's update date.
            Date updatedAt = new Date(0);
            String version = "";
            for (String completeKey : dependencies)
            {
                Bundle superbatch = bundles.get(completeKey);
                if (superbatch != null)
                {
                    if (superbatch.getUpdatedAt().after(updatedAt))
                    {
                        updatedAt = superbatch.getUpdatedAt();
                    }
                    version = HashBuilder.buildHash(version, superbatch.getVersion());
                }
            }
            String superbatchKey = CONTEXT_RESOURCE_PREFIX + ":" + DefaultResourceDependencyResolver
                    .IMPLICIT_CONTEXT_NAME;
            bundles.put(superbatchKey, new Bundle(snapshot, superbatchKey, dependencies, updatedAt, version, true));
        }

        ensureNoLegacyStuff(snapshot);
        return bundles;
    }

    /**
     * Queries the plugin system and return list of Resources for Web Resource.
     */
    public Map<String, Resource> getResourcesWithoutCache(Bundle bundle)
    {
        String completeKey = bundle.getKey();
        ModuleDescriptor<?> moduleDescriptor = integration.getPluginAccessor().getEnabledPluginModule(completeKey);
        // In case of virtual context or super batch resource there would be null returned.
        if (moduleDescriptor == null)
        {
            return new HashMap<String, Resource>();
        }
        if (!(moduleDescriptor instanceof WebResourceModuleDescriptor))
        {
            throw new RuntimeException("module " + completeKey + "isn't the web resource!");
        }
        WebResourceModuleDescriptor webResourceDescriptor = (WebResourceModuleDescriptor) moduleDescriptor;

        // Adding resources, it is important to use `LinkedHashMap` because order of resources is important.
        Map<String, Resource> resources = new LinkedHashMap<String, Resource>();
        for (ResourceDescriptor resourceDescriptor : webResourceDescriptor.getResourceDescriptors())
        {
            if (RESOURCE_DOWNLOAD_TYPE.equals(resourceDescriptor.getType()))
            {
                ResourceLocation resourceLocation = resourceDescriptor.getResourceLocationForName(null);
                Resource resource = buildResource(bundle, resourceLocation);
                resources.put(resource.getName(), resource);
            }
        }
        return resources;
    }

    /**
     * Returns content of Resource, it could be Resource of Web Resource or Resource of Plugin.
     *
     * @return content of Resource.
     */
    public Content getContentFor(Resource resource)
    {
        Plugin plugin;
        ResourceLocation resourceLocation = resource.getResourceLocation();
        boolean isMinificationEnabled;

        // Detecting if it's a web resource key or plugin key.
        if (resource.getKey().contains(":"))
        {
            ModuleDescriptor<?> moduleDescriptor = integration.getPluginAccessor().getEnabledPluginModule(resource
                    .getKey());
            plugin = checkNotNull(moduleDescriptor.getPlugin());
            if (moduleDescriptor instanceof WebResourceModuleDescriptor)
            {
                isMinificationEnabled = !((WebResourceModuleDescriptor) moduleDescriptor).isDisableMinification();
            }
            else
            {
                isMinificationEnabled = true;
            }
        }
        else
        {
            plugin = checkNotNull(integration.getPluginAccessor().getEnabledPlugin(resource.getKey()));
            isMinificationEnabled = true;
        }

        String sourceParam = resourceLocation.getParameter(RESOURCE_SOURCE_PARAM);
        final DownloadableResource downloadableResource;
        if ("webContextStatic".equalsIgnoreCase(sourceParam))
        {
            downloadableResource = new DownloadableWebResource(plugin, resourceLocation, resource.getFilePath(),
                    servletContextFactory.getServletContext(), !isMinificationEnabled);
        }
        else
        {
            downloadableResource = new DownloadableClasspathResource(plugin, resourceLocation, resource.getFilePath());
        }

        return new ContentImpl(resource.getContentType(), false)
        {
            @Override
            public SourceMap writeTo(OutputStream out, boolean isSourceMapEnabled)
            {
                try
                {
                    downloadableResource.streamResource(out);
                    return null;
                }
                catch (DownloadException e)
                {
                    throw new RuntimeException(e);
                }
            }
        };
    }

    /**
     * Helper to hide bunch of integration code and simplify resource creation.
     */
    protected Resource buildResource(Bundle bundle, ResourceLocation resourceLocation)
    {
        // The code based on `SingleDownloadableResourceFinder.getDownloadablePluginResource`.
        final String sourceParam = resourceLocation.getParameter(RESOURCE_SOURCE_PARAM);
        String type = ResourceUtils.getType(!StringUtils.isEmpty(resourceLocation.getName()) ? resourceLocation
                .getName() : resourceLocation.getName());
        boolean isRedirect = "webContext".equalsIgnoreCase(sourceParam);
        boolean isBatchable = !isRedirect && !"false".equalsIgnoreCase(resourceLocation.getParameter
                (RESOURCE_BATCH_PARAM));
        boolean isCacheable = !"false".equalsIgnoreCase(resourceLocation.getParams().get("cache"));

        Resource resource = new Resource(bundle, resourceLocation, type, isBatchable, isRedirect, isCacheable);
        return resource;
    }

    /**
     * Get not declared Resource.
     */
    public Resource getModuleResource(String completeKey, String name)
    {
        // Confluence throws error if trying to query module with not complete key (with the plugin key for example).
        if (!isWebResourceKey(completeKey))
        {
            return null;
        }
        ModuleDescriptor<?> moduleDescriptor = integration.getPluginAccessor().getEnabledPluginModule(completeKey);
        // In case of virtual context or super batch resource there would be null returned.
        if (moduleDescriptor == null)
        {
            return null;
        }
        ResourceLocation resourceLocation = moduleDescriptor.getResourceLocation(RESOURCE_DOWNLOAD_TYPE, name);
        if (resourceLocation == null)
        {
            return null;
        }
        Plugin plugin = moduleDescriptor.getPlugin();
        Date updatedAt = (plugin.getDateLoaded() == null) ? new Date() : plugin.getDateLoaded();
        PluginResourceContainer resourceContainer = new PluginResourceContainer(new Snapshot(this), completeKey
            , updatedAt, plugin.getPluginInformation().getVersion());
        return buildResource(resourceContainer, resourceLocation);
    }

    /**
     * Get Resource for Plugin.
     */
    public Resource getPluginResource(String pluginKey, String name)
    {
        Plugin plugin = integration.getPluginAccessor().getPlugin(pluginKey);
        if (plugin == null)
        {
            return null;
        }
        ResourceLocation resourceLocation = plugin.getResourceLocation(RESOURCE_DOWNLOAD_TYPE, name);
        if (resourceLocation == null)
        {
            return null;
        }
        PluginResourceContainer resourceContainer = new PluginResourceContainer(new Snapshot(this), pluginKey,
                plugin.getDateLoaded(), plugin.getPluginInformation().getVersion());
        Resource resource = buildResource(resourceContainer, resourceLocation);
        return resource;
    }

    public static boolean couldBeVirtualContext(String virtualContextKey)
    {
        return virtualContextKey.startsWith(CONTEXT_RESOURCE_PREFIX + ":");
    }

    public static String virtualContextKeyToWebResourceKey(String virtualContextKey)
    {
        return virtualContextKey.replace(CONTEXT_RESOURCE_PREFIX + ":", "");
    }

    /**
     * If `completeKey` is Web Resource key or Plugin key.
     */
    public static boolean isWebResourceKey(String completeKey)
    {
        return completeKey.contains(":");
    }

    /**
     * Hash code representing the state of config. State of the system could change not only on plugin system changes
     * but also some properties of config could change too. In such cases the cache additionally checked against the
     * version of config it's build for.
     * <p/>
     * Strictly speaking this hash should include the full state of config, but it would be too slow, so instead we
     * check only for some of its properties.
     */
    public int partialHashCode()
    {
        return integration.getSuperBatchVersion().hashCode();
    }

    @Deprecated
    public WebResourceIntegration getIntegration()
    {
        return integration;
    }

    @Deprecated
    TransformerCache getTransformerCache()
    {
        return transformerCache;
    }
}