package com.atlassian.plugin.webresource;

import java.util.*;

import com.atlassian.plugin.webresource.url.UrlParameters;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ListMultimap;
import org.apache.commons.collections.CollectionUtils;

import com.google.common.collect.Iterables;

/**
 * A helper classes that knows how to perform certain operations on ContextBatch beans.
 */
class ContextBatchOperations
{
    static final String CONTEXT_SEPARATOR = ",";
    static final String CONTEXT_SUBTRACTION = "-";
        
    private PluginResourceLocator pluginResourceLocator;
    private final WebResourceFilter filter;
    
    /**
     * Parse a context batch key which typically has the form batch1,batch2,-excludedBatch3,-excludedBatch4
     * 
     * @param key the key to parse
     * @param included the ordered Set to be populated with included batch names.
     * @param excluded the Set to be populated with excluded batch names. The order of exclusion is not important.
     */
    static void parseContexts(final String key, LinkedHashSet<String> included, Set<String> excluded)
    {
        String[] split = key.split(CONTEXT_SEPARATOR);
        for (String s : split)
        {
            if (s.startsWith(CONTEXT_SUBTRACTION))
            {
                excluded.add(s.substring(1));
            }
            else
            {
                included.add(s);
            }
        }
    }    
    
    ContextBatchOperations(PluginResourceLocator pluginResourceLocator, WebResourceFilter filter)
    {
        this.pluginResourceLocator = pluginResourceLocator;
        this.filter = filter;
    }
    
    /**
     * Merges context batches into a single context batch. 
     * <p/>
     * <strong>Note:</strong>you can only merge batches if all batches <strong>do not</strong> 
     * have any excluded contexts. The problem is that with excluded contexts you can't know
     * if the batch you are merging with should have resources (and resource parameters)
     * removed and you are therefore going to end up with a merged batch with a potentially
     * wrong hash. (See {@link ContextBatch#createHash()})
     * 
     * @param batchesToMerge - one or more batches to merge together 
     * @return a single context batch which is the result of merging all the supplied batches.
     * @throw IllegalArgumentException if any of the batches have excludedContexts
     */
    ContextBatch merge(final Collection<ContextBatch> batchesToMerge)
    {
        if (CollectionUtils.isEmpty(batchesToMerge))
            return null;
        
        if (batchesToMerge.size() == 1)
            return batchesToMerge.iterator().next();
        
        final StringBuilder mergedKey = new StringBuilder();
        // the ordering of contexts must be maintained for consistency of resources within the batch
        final LinkedHashSet<String> includedContexts = new LinkedHashSet<String>();
        final Set<WebResourceModuleDescriptor> resources = new HashSet<WebResourceModuleDescriptor>();
        final ListMultimap<PluginResourceBatchParams, UrlParameters> batchResourceParams = ArrayListMultimap.create();

        boolean removeSuperResources = false;
        for (ContextBatch batch : batchesToMerge)
        {
            if (!Iterables.isEmpty(batch.getExcludedContexts()))
                throw new IllegalArgumentException("The ContextBatch " + batch.getKey() + " has excludedContexts.");
            removeSuperResources |= batch.isRemoveSuperResources();
            mergedKey.append(batch.getKey()).append(CONTEXT_SEPARATOR);
            includedContexts.addAll(batch.getContexts());
            Iterables.addAll(resources, batch.getResources());
            batchResourceParams.putAll(batch.getResourceParams());
        }
        
        mergedKey.deleteCharAt(mergedKey.length() - 1);

        return new ContextBatch(mergedKey.toString(), new ArrayList<String>(includedContexts), null, resources, batchResourceParams, removeSuperResources);
    }
    
    /**
     * Subtract ContextBatches from the supplied operand, creating a new ContextBatch (unless there are no
     * batches to subtract in which case the supplied operand is returned unchanged).
     * <p/>
     * Subtraction for a ContextBatch means the removal of any WebResourceModuleDescriptor that exist within
     * the ContextBatch being subtracted. Consequently it also means that some PluginResourceBatchParams
     * may also be removed (if there is no longer an applicable PluginResource). 
     * <p/> 
     * <strong>Note:</strong>you can only subtract batches if all batches <strong>do not</strong> 
     * have any excluded contexts. The problem is that with excluded contexts you end up with the
     * possibility of subtracting a subtracted context. And since the already subtracted context no
     * longer has reference to the WebResourceModuleDescriptor it caused to be removed you cannot
     * ensure that the resultant ContextBatch also has them removed.
     * 
     * @param operand the ContextBatch to be operated upon
     * @param batchesToSubtract the ContextBatches to be subtracted.
     * @return the ContextBatch resulting in subtracting the batchesToSubtract.
     * @throw IllegalArgumentException if any of the batchesToSubtract already have excluded contexts.
     */
    ContextBatch subtract(final ContextBatch operand, final Collection<ContextBatch> batchesToSubtract)
    {
        if (CollectionUtils.isEmpty(batchesToSubtract))
            return operand;

        // use linked hash map: preserve a consistent deterministic ordering irrespective of hash algorithm
        Collection<String> excludedContexts = new LinkedHashSet<String>();
        Iterables.addAll(excludedContexts, operand.getExcludedContexts());
        final Collection<WebResourceModuleDescriptor> resources = new LinkedHashSet<WebResourceModuleDescriptor>();
        Iterables.addAll(resources, operand.getResources());

        boolean removeSuperResources = false;
        for (ContextBatch subtract : batchesToSubtract)
        {
            if (!Iterables.isEmpty(subtract.getExcludedContexts()))
            {
                throw new IllegalArgumentException("The ContextBatch " + subtract.getKey() + " has excludedContexts.");
            }

            removeSuperResources |= subtract.isRemoveSuperResources();
            
            Iterables.addAll(excludedContexts, subtract.getContexts());
            Iterable<WebResourceModuleDescriptor> subtractResources = subtract.getResources();
            for (WebResourceModuleDescriptor resource : subtractResources)
            {
                resources.remove(resource);
            }
        }

        String key = buildContextKey(operand.getContexts(), excludedContexts);
        ContextBatch subtractionResult = new ContextBatch(key, operand.getContexts(),excludedContexts, resources, null, removeSuperResources);
        
        // now calculate the new PluginResourceBatchParams for the remaining WebResourceModuleDescriptors
        for (WebResourceModuleDescriptor resource : resources)
        {
            for (final PluginResource pluginResource : pluginResourceLocator.getPluginResources(resource.getCompleteKey()))
            {
                if (filter.matches(pluginResource.getResourceName()))
                {
                    subtractionResult.addResourceType(pluginResource);
                }
            }
        }
        
        return subtractionResult;
    }

    static String buildContextKey(Iterable<String> contexts, Iterable<String> excludedContexts) {
        StringBuilder key = new StringBuilder();
        for (String includedContext : contexts)
        {
            key.append(includedContext).append(CONTEXT_SEPARATOR);
        }

        for (String excludedContext : excludedContexts)
        {
            key.append(CONTEXT_SUBTRACTION).append(excludedContext).append(CONTEXT_SEPARATOR);
        }

        key.deleteCharAt(key.length() - 1);
        return key.toString();
    }
}
