package com.atlassian.plugin.webresource;

import com.atlassian.plugin.webresource.url.CDNSupport;
import com.atlassian.plugin.webresource.url.NamedHash;
import com.atlassian.plugin.webresource.url.UrlParameters;
import com.atlassian.plugin.webresource.util.HashBuilder;
import com.google.common.base.Function;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Abstract base class for plugin resources
 * @since v3.0
 */
public abstract class AbstractPluginResource implements PluginResource
{
    private static final Logger logger = LoggerFactory.getLogger(AbstractPluginResource.class);

    protected final UrlParameters urlParams;
    protected final Map<String, String> params;
    protected final Set<BatchedWebResourceDescriptor> batchedWebResourceDescriptors;

    protected AbstractPluginResource(UrlParameters urlParams, Map<String, String> params,
        Set<BatchedWebResourceDescriptor> batchedWebResourceDescriptors)
    {
        this.urlParams = urlParams;
        this.params = ImmutableMap.copyOf(params);
        this.batchedWebResourceDescriptors = batchedWebResourceDescriptors;
    }

    @Override
    public UrlParameters getUrlParameters()
    {
        return urlParams;
    }

    @Override
    public CDNSupport supportsCdn()
    {
        return urlParams.supportsCdn();
    }

    @Override
    public List<NamedHash> getHashes()
    {
        return urlParams.allHashes();
    }

    @Override
    public Map<String, String> getParams()
    {
        return params;
    }

    @Override
    public Map<String, String> getBatchableParams()
    {
        return urlParams.toQueryString();
    }

    @Override
    public Set<BatchedWebResourceDescriptor> getBatchedWebResourceDescriptors()
    {
        return batchedWebResourceDescriptors;
    }

    @Override
    public String getStaticUrl(WebResourceIntegration webResourceIntegration, WebResourceUrlProvider webResourceUrlProvider, UrlMode urlMode)
    {
        final String url;
        if (isCacheSupported())
        {
            List<String> hashes = Lists.transform(getHashes(), new Function<NamedHash, String>()
            {
                @Override
                public String apply(@Nullable NamedHash namedHash)
                {
                    return namedHash.getHash();
                }
            });
            String contributedHash = HashBuilder.buildHash(hashes) + supportsCdn().hashSuffix();
            if (supportsCdn(webResourceIntegration))
            {
                String relativeUrl = webResourceUrlProvider.getStaticResourcePrefix(contributedHash, getVersion(webResourceIntegration), UrlMode.RELATIVE)
                        + getUrl();
                url = webResourceIntegration.getCDNStrategy().transformRelativeUrl(relativeUrl);
            }
            else
            {
                url = webResourceUrlProvider.getStaticResourcePrefix(contributedHash, getVersion(webResourceIntegration), urlMode)
                    + getUrl();
            }
        }
        else
        {
            url = webResourceUrlProvider.getBaseUrl(urlMode) + getUrl();
        }

        if (logger.isDebugEnabled())
        {
            logger.debug("Static hashes for url: {}", getUrl());
            for (NamedHash namedHash : getHashes())
            {
                logger.debug("{}: {}", namedHash.getName(), namedHash.getHash());
            }
        }
        return url;
    }

    private boolean supportsCdn(WebResourceIntegration webResourceIntegration)
    {
        return CDNSupport.OK.equals(supportsCdn())
                && null != webResourceIntegration.getCDNStrategy()
                && webResourceIntegration.getCDNStrategy().supportsCdn();
    }
}
