package com.atlassian.plugin.webresource;

import com.atlassian.plugin.cache.filecache.FileCacheStreamProvider;
import com.atlassian.plugin.servlet.DownloadException;
import com.atlassian.plugin.servlet.DownloadableResource;
import com.atlassian.plugin.webresource.cache.CacheHandle;
import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Map;

import static com.google.common.collect.Iterables.any;

/**
 * A base class that batches together a sequence of DownloadableResources
 *
 * @since 2.13
 */
public abstract class AbstractBatchDownloadableResource implements DownloadableResource
{
    private static final Logger log = LoggerFactory.getLogger(AbstractBatchDownloadableResource.class);

    private static final NewlineResourceContentAnnotator newlineResourceContentAnnotator = new NewlineResourceContentAnnotator();
    private static final SemicolonResourceContentAnnotator semicolonResourceContentAnnotator = new SemicolonResourceContentAnnotator();
    private static final TryCatchJsResourceContentAnnotator tryCatchJsResourceContentAnnotator = new TryCatchJsResourceContentAnnotator();
    
    private static final ResourceContentAnnotator[] DEFAULT_ANNOTATORS = new ResourceContentAnnotator[] {
        newlineResourceContentAnnotator
    };

    private static final ResourceContentAnnotator[] DEFAULT_JS_ANNOTATORS = new ResourceContentAnnotator[] {
        newlineResourceContentAnnotator,
        semicolonResourceContentAnnotator
    };
    
    private static final ResourceContentAnnotator[] JS_WRAP_ANNOTATORS = new ResourceContentAnnotator[] {
        newlineResourceContentAnnotator,
        semicolonResourceContentAnnotator,
        tryCatchJsResourceContentAnnotator
    };
    
    private final String type;
    private final Map<String, String> params;
    private final Iterable<DownloadableResource> resources;
    private final CacheHandle cacher;
    private final ResourceContentAnnotator[] resourceContentAnnotators;

    /**
     * This constructor should only ever be used internally within this class. It does not ensure that the resourceName's
     * file extension is the same as the given type. It is up to the calling code to ensure this.
     * @param type - the type of resource ("css", "js")
     * @param params - the parameters of the resource (ieonly, media, etc)
     * @param resources - the resources included in the batch.
     * @param cacher
     * @param resourceBatchingConfiguration the configuration for the batching behaviour
     */
    AbstractBatchDownloadableResource(final String type, final Map<String, String> params,
                                      final Iterable<DownloadableResource> resources, final CacheHandle cacher, 
                                      final ResourceBatchingConfiguration resourceBatchingConfiguration)
    {
        this.type = type;
        this.params = ImmutableMap.copyOf(params);
        this.resources = resources;
        this.cacher = cacher;
        this.resourceContentAnnotators = getAnnotators(resourceBatchingConfiguration); 
    }

    /**
     * @return true if there are no resources included in this batch
     */
    public boolean isEmpty()
    {
        return Iterables.isEmpty(resources);
    }

    public boolean isResourceModified(final HttpServletRequest request, final HttpServletResponse response)
    {
        return any(resources, new Predicate<DownloadableResource>()
        {
            public boolean apply(final DownloadableResource resource)
            {
                return resource.isResourceModified(request, response);
            }
        });
    }

    public void serveResource(final HttpServletRequest request, final HttpServletResponse response) throws DownloadException
    {
        log.debug("Start to serve batch {}", this);

        OutputStream out;
        try
        {
            out = response.getOutputStream();
        }
        catch (final IOException e)
        {
            throw new DownloadException(e);
        }

        streamResourceInternal(out, resourceContentAnnotators);

        log.debug("Finished serving batch {}", this);
    }

    public void streamResource(final OutputStream originalOut) throws DownloadException
    {
        streamResourceInternal(originalOut,resourceContentAnnotators);
    }
    
    private void streamResourceInternal(final OutputStream originalOut, final ResourceContentAnnotator[] annotators) throws DownloadException
    {
        final FileCacheStreamProvider streamProvider = new FileCacheStreamProvider() {
            @Override
            public void writeStream(OutputStream dest) throws DownloadException {
                for (final DownloadableResource resource : resources)
                {
                    try
                    {
                        applyBeforeAnnotators(dest, annotators);                    
                        resource.streamResource(dest);
                        applyAfterAnnotators(dest, annotators);
                    }
                    catch (IOException ex)
                    {
                        throw new DownloadException(ex);
                    }
                }
            }
        };
        cacher.stream(originalOut, streamProvider);
    }

    public String getContentType()
    {
        final String contentType = params.get("content-type");
        if (contentType != null)
        {
            return contentType;
        }
        return null;
    }

    public Map<String, String> getParams()
    {
        return params;
    }

    public String getType()
    {
        return type;
    }
    
    private void applyBeforeAnnotators(OutputStream str, ResourceContentAnnotator[] annotators) throws IOException
    {
        for (ResourceContentAnnotator annotator : annotators)
        {
            log.debug("Applying before annotator {}", annotator.getClass().getSimpleName());
            annotator.before(str);
            log.debug("Finished applying before annotator {}", annotator.getClass().getSimpleName());
        }
    }
    
    /**
     * Apply the after annotators in reverse order.
     * 
     * @param str
     * @throws IOException
     */
    private void applyAfterAnnotators(OutputStream str, ResourceContentAnnotator[] annotators) throws IOException
    {
        for (int i = annotators.length - 1; i >= 0; i--)
        {
            log.debug("Applying after annotator {}", annotators[i].getClass().getSimpleName());
            annotators[i].after(str);
            log.debug("Finished applying after annotator {}", annotators[i].getClass().getSimpleName());
        }
    }
    
    /**
     * 
     * @param batchingConfig the resource batching configuration
     * @return the annotators to use when writing the requested Batch resource.
     */
    private ResourceContentAnnotator[] getAnnotators(ResourceBatchingConfiguration batchingConfig) 
    {
        if ("js".equals(this.getType()))
        {
            if (batchingConfig.isJavaScriptTryCatchWrappingEnabled())
            {
                return JS_WRAP_ANNOTATORS;
            }
            else
            {
                return DEFAULT_JS_ANNOTATORS;
            }
        }
        else
        {
            return DEFAULT_ANNOTATORS;
        }
    }
}
