package com.atlassian.plugin.webresource;

import static com.atlassian.plugin.servlet.AbstractFileServerServlet.PATH_SEPARATOR;
import static com.atlassian.plugin.util.EfficientStringUtils.endsWith;
import static com.atlassian.plugin.webresource.ContextBatchPluginResource.URL_PREFIX;
import static com.atlassian.plugin.webresource.SuperBatchPluginResource.DEFAULT_RESOURCE_NAME_PREFIX;
import static com.google.common.collect.Sets.newHashSet;

import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.cache.filecache.FileCache;
import com.atlassian.plugin.servlet.DownloadableResource;

import com.atlassian.plugin.webresource.cache.CacheHandle;
import com.atlassian.plugin.webresource.cache.FileCacheKey;
import com.google.common.collect.Iterables;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Constructs a context batch resource for download
 * 
 * @since 2.9.0
 */
class ContextBatchDownloadableResourceBuilder extends AbstractBatchResourceBuilder
{
    private final ResourceDependencyResolver dependencyResolver;

    ContextBatchDownloadableResourceBuilder(final ResourceDependencyResolver dependencyResolver, final PluginAccessor pluginAccessor, final WebResourceUrlProvider webResourceUrlProvider,
                                            final DownloadableResourceFinder resourceFinder, final FileCache<FileCacheKey> cache,
                                            ResourceBatchingConfiguration resourceBatchingConfiguration)
    {
        super(pluginAccessor, webResourceUrlProvider, resourceFinder, cache, resourceBatchingConfiguration);
        this.dependencyResolver = dependencyResolver;
    }

    public boolean matches(final String path)
    {
        final String type = ResourceUtils.getType(path);
        return (path.indexOf(URL_PREFIX + type) > -1) && endsWith(path, DEFAULT_RESOURCE_NAME_PREFIX, ".", type);
    }

    public ContextBatchDownloadableResource parse(final String path, final Map<String, String> params)
    {
        final String type = ResourceUtils.getType(path);
        final String key = getKey(path);
        final LinkedHashSet<String> includedContexts = new LinkedHashSet<String>();
        final Set<String> excludedContexts = new HashSet<String>();

        ContextBatchOperations.parseContexts(key, includedContexts, excludedContexts);

        LinkedHashSet<WebResourceModuleDescriptor> moduleDescriptors = new LinkedHashSet<WebResourceModuleDescriptor>();

        for (final String context : includedContexts)
        {
            Iterables.addAll(moduleDescriptors, dependencyResolver.getDependenciesInContext(context));
        }

        for (final String context : excludedContexts)
        {
            List<WebResourceModuleDescriptor> dependencies = new ArrayList<WebResourceModuleDescriptor>();
            Iterables.addAll(dependencies, dependencyResolver.getDependenciesInContext(context));

            moduleDescriptors.removeAll(dependencies);
        }

        final Set<String> alreadyIncluded = newHashSet();
        List<DownloadableResource> resources = new ArrayList<DownloadableResource>();
        for (final WebResourceModuleDescriptor moduleDescriptor : moduleDescriptors)
        {
            String moduleKey = moduleDescriptor.getCompleteKey();
            if (!alreadyIncluded.contains(moduleKey))
            {
                Iterables.addAll(resources, resolve(moduleDescriptor, type, params));
                alreadyIncluded.add(moduleKey);
            }
        }
        final CacheHandle cacher = CacheHandle.Builder.forRequest(cache, "context", path, params);
        return new ContextBatchDownloadableResource(type, params, resources, cacher, resourceBatchingConfiguration);
    }

    private String getKey(final String path)
    {
        final int secondSlashIndex = path.lastIndexOf(PATH_SEPARATOR);
        final int firstSlashIndex = path.lastIndexOf(PATH_SEPARATOR, secondSlashIndex - 1);
        return path.substring(firstSlashIndex + 1, secondSlashIndex);
    }
}
