package com.atlassian.plugin.webresource.assembler;

import com.atlassian.json.marshal.Jsonable;
import com.atlassian.json.marshal.wrapped.JsonableBoolean;
import com.atlassian.json.marshal.wrapped.JsonableNumber;
import com.atlassian.json.marshal.wrapped.JsonableString;
import com.atlassian.plugin.webresource.ContextBatchBuilder;
import com.atlassian.plugin.webresource.DefaultResourceDependencyResolver;
import com.atlassian.plugin.webresource.DefaultWebResourceFilter;
import com.atlassian.plugin.webresource.PluginResource;
import com.atlassian.plugin.webresource.PluginResourceLocator;
import com.atlassian.plugin.webresource.ResourceBatchingConfiguration;
import com.atlassian.plugin.webresource.ResourceDependencyResolver;
import com.atlassian.plugin.webresource.SuperBatchBuilder;
import com.atlassian.plugin.webresource.TransformDescriptorToKey;
import com.atlassian.plugin.webresource.WebResourceFilter;
import com.atlassian.plugin.webresource.WebResourceIntegration;
import com.atlassian.plugin.webresource.WebResourceModuleDescriptor;
import com.atlassian.plugin.webresource.WebResourceUrlProvider;
import com.atlassian.plugin.webresource.data.DefaultPluginDataResource;
import com.atlassian.webresource.api.assembler.AssembledResources;
import com.atlassian.webresource.api.assembler.RequiredData;
import com.atlassian.webresource.api.assembler.RequiredResources;
import com.atlassian.webresource.api.assembler.WebResourceAssembler;
import com.atlassian.webresource.api.assembler.WebResourceSet;
import com.atlassian.webresource.api.data.PluginDataResource;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.google.common.collect.Iterables.transform;

/**
 * Implementation of WebResourceAssembler.
 * @since v3.0
 */
class DefaultWebResourceAssembler implements WebResourceAssembler
{
    /** Webresources to be included on next call to includeResources */
    private final Set<String> webResourcesToInclude;
    /** Webresource contexts to be included on next call to includeResources */
    private final Set<String> contextsToInclude;
    /** Data to be included on the next call to includeResources */
    private final Map<String, Jsonable> dataToInclude;

    private final AssembledResources assembledResources;
    private final RequiredResources requiredResources;
    private final RequiredData requiredData;
    private final ResourceRequirer resourceRequirer;

    /** Current state of previously-included resources */
    private InclusionState inclusion;

    private static class InclusionState
    {
        /** Has the superbatch been included */
        private boolean superbatch;
        /** Webresources that have been included in previous calls to includeResources, and all the individual resources
         * in included contexts */
        private Set<String> webresources;
        /** Webresource contexts that have been included in previous calls to includeResources */
        private Set<String> contexts;
        /** Data keys that have been included in previous calls to includeData */
        private Set<String> dataKeys;

        public InclusionState(boolean superbatch, Set<String> webresources, Set<String> contexts, Set<String> dataKeys)
        {
            this.superbatch = superbatch;
            this.webresources = webresources;
            this.contexts = contexts;
            this.dataKeys = dataKeys;
        }

        public InclusionState copy()
        {
            return new InclusionState(superbatch, Sets.newHashSet(webresources), Sets.newHashSet(contexts),
                    Sets.newHashSet(dataKeys));
        }
    }

    public DefaultWebResourceAssembler(ResourceBatchingConfiguration batchingConfiguration,
           WebResourceIntegration webResourceIntegration, PluginResourceLocator pluginResourceLocator,
           WebResourceUrlProvider webResourceUrlProvider)
    {
        this(batchingConfiguration, webResourceIntegration, pluginResourceLocator, webResourceUrlProvider,
                Sets.<String>newLinkedHashSet(), Sets.<String>newLinkedHashSet(), Maps.<String, Jsonable>newLinkedHashMap(),
                new InclusionState(false, Sets.<String>newHashSet(), Sets.<String>newHashSet(), Sets.<String>newHashSet()));
    }

    private DefaultWebResourceAssembler(ResourceBatchingConfiguration batchingConfiguration,
        WebResourceIntegration webResourceIntegration, PluginResourceLocator pluginResourceLocator,
        WebResourceUrlProvider webResourceUrlProvider, Set<String> webResourcesToInclude, Set<String> contextsToInclude,
        Map<String, Jsonable> dataToInclude, InclusionState inclusionState)
    {
        this.resourceRequirer = new ResourceRequirer(batchingConfiguration, webResourceIntegration,
                webResourceUrlProvider, pluginResourceLocator,
                new DefaultResourceDependencyResolver(webResourceIntegration, batchingConfiguration));
        this.webResourcesToInclude = webResourcesToInclude;
        this.contextsToInclude = contextsToInclude;
        this.dataToInclude = dataToInclude;
        inclusion = inclusionState;
        assembledResources = new DefaultAssembledResources();
        requiredResources = new DefaultRequiredResources();
        requiredData = new DefaultRequiredData();
    }

    @Override
    public AssembledResources assembled()
    {
        return assembledResources;
    }

    @Override
    public RequiredResources resources()
    {
        return requiredResources;
    }

    @Override
    public RequiredData data()
    {
        return requiredData;
    }

    @Override
    public WebResourceAssembler copy()
    {
        return new DefaultWebResourceAssembler(resourceRequirer.batchingConfiguration,
                resourceRequirer.webResourceIntegration, resourceRequirer.pluginResourceLocator, resourceRequirer.webResourceUrlProvider,
                Sets.newLinkedHashSet(webResourcesToInclude), Sets.newLinkedHashSet(contextsToInclude), Maps.newLinkedHashMap(dataToInclude),
                inclusion.copy());
    }

    private final class DefaultRequiredResources implements RequiredResources
    {
        @Override
        public RequiredResources requireWebResource(String moduleCompleteKey)
        {
            webResourcesToInclude.add(moduleCompleteKey);
            return this;
        }

        @Override
        public RequiredResources requireContext(String context)
        {
            contextsToInclude.add(context);
            return this;
        }

        @Override
        public RequiredResources exclude(Set<String> webResources, Set<String> contexts)
        {
            InclusionState nextInclusion = inclusion.copy();
            Set<String> excludeWebResources = null == webResources ? Collections.<String>emptySet() : Sets.newHashSet(webResources);
            Set<String> excludeContexts = null == contexts ? Collections.<String>emptySet() : Sets.newHashSet(contexts);
            resourceRequirer.includeResources(excludeWebResources, excludeContexts,
                    Collections.<String, Jsonable>emptyMap(), DefaultWebResourceFilter.INSTANCE, nextInclusion);
            inclusion = nextInclusion;
            return this;
        }
    }

    private final class DefaultRequiredData implements RequiredData
    {
        @Override
        public RequiredData requireData(String key, Jsonable content)
        {
            dataToInclude.put(key, content);
            return this;
        }

        @Override
        public RequiredData requireData(String key, Number content)
        {
            dataToInclude.put(key, new JsonableNumber(content));
            return this;
        }

        @Override
        public RequiredData requireData(String key, String content)
        {
            dataToInclude.put(key, new JsonableString(content));
            return this;
        }

        @Override
        public RequiredData requireData(String key, Boolean content)
        {
            dataToInclude.put(key, new JsonableBoolean(content));
            return this;
        }
    }

    private final class DefaultAssembledResources implements AssembledResources
    {
        @Override
        public WebResourceSet drainIncludedResources()
        {
            InclusionState nextInclusion = inclusion.copy();
            WebResourceSet webResourceSet = resourceRequirer.includeResources(webResourcesToInclude, contextsToInclude,
                    dataToInclude, DefaultWebResourceFilter.INSTANCE, nextInclusion);
            webResourcesToInclude.clear();
            contextsToInclude.clear();
            dataToInclude.clear();
            inclusion = nextInclusion;
            return webResourceSet;
        }

        @Override
        public WebResourceSet peek()
        {
            // Copy inclusion state - it's not modified by this operation
            InclusionState fakeInclusion = inclusion.copy();
            return resourceRequirer.includeResources(webResourcesToInclude, contextsToInclude, dataToInclude,
                    DefaultWebResourceFilter.INSTANCE, fakeInclusion);
        }
    }

    private static class ResourceRequirer
    {
        private final ResourceBatchingConfiguration batchingConfiguration;
        private final WebResourceIntegration webResourceIntegration;
        private final WebResourceUrlProvider webResourceUrlProvider;
        private final PluginResourceLocator pluginResourceLocator;
        private final ResourceDependencyResolver dependencyResolver;

        private ResourceRequirer(ResourceBatchingConfiguration batchingConfiguration, WebResourceIntegration webResourceIntegration,
                                 WebResourceUrlProvider webResourceUrlProvider, PluginResourceLocator pluginResourceLocator,
                                 ResourceDependencyResolver dependencyResolver)
        {
            this.batchingConfiguration = batchingConfiguration;
            this.webResourceIntegration = webResourceIntegration;
            this.webResourceUrlProvider = webResourceUrlProvider;
            this.pluginResourceLocator = pluginResourceLocator;
            this.dependencyResolver = dependencyResolver;
        }

        // Collection of PluginResource + PluginDataResources
        private static class Resources
        {
            private final Collection<PluginResource> resources;
            private final Collection<PluginDataResource> data;

            private Resources()
            {
                this.resources = Lists.newLinkedList();
                this.data = Lists.newLinkedList();
            }
        }

        private WebResourceSet includeResources(Set<String> requiredWebResources, Set<String> requiredContexts,
            Map<String, Jsonable> requiredData, WebResourceFilter filter, InclusionState inclusion)
        {
            final Resources resourcesToInclude = new Resources();

            // Add superbatch
            addSuperBatchResources(resourcesToInclude, filter, inclusion);

            // Add contexts
            addContextBatchDependencies(resourcesToInclude, filter, requiredWebResources, requiredContexts, inclusion);

            // Add webresources
            Iterable<String> dependencyModuleKeys = getAllModuleKeysDependencies(requiredWebResources);
            addModuleResources(resourcesToInclude, dependencyModuleKeys, inclusion.webresources, filter);

            for (Map.Entry<String, Jsonable> data : requiredData.entrySet())
            {
                if (!inclusion.dataKeys.contains(data.getKey()))
                {
                    resourcesToInclude.data.add(new DefaultPluginDataResource(data.getKey(), data.getValue()));
                }
            }

            // Add contexts, webresources & data to inclusion state
            Iterables.addAll(inclusion.contexts, requiredContexts);
            Iterables.addAll(inclusion.webresources, dependencyModuleKeys);
            Iterables.addAll(inclusion.dataKeys, requiredData.keySet());

            return new DefaultWebResourceSet(batchingConfiguration, webResourceUrlProvider,
                    webResourceIntegration, resourcesToInclude.resources, resourcesToInclude.data);
        }

        private void addSuperBatchResources(Resources resourcesToInclude, WebResourceFilter filter, InclusionState inclusion)
        {
            if (inclusion.superbatch || !batchingConfiguration.isSuperBatchingEnabled())
            {
                return;
            }
            inclusion.superbatch = true;
            Iterables.addAll(resourcesToInclude.resources, new SuperBatchBuilder(dependencyResolver, pluginResourceLocator, webResourceIntegration).build(filter));
            for (WebResourceModuleDescriptor wrmd : dependencyResolver.getSuperBatchDependencies())
            {
                Iterables.addAll(resourcesToInclude.data, pluginResourceLocator.getPluginDataResources(wrmd.getCompleteKey()));
            }
        }

        private void addContextBatchDependencies(Resources resourcesToInclude, WebResourceFilter filter,
             Set<String> requiredWebResources, Set<String> requiredContexts, InclusionState inclusion)
        {
            final ContextBatchBuilder builder = new ContextBatchBuilder(webResourceIntegration.getPluginAccessor(), pluginResourceLocator, dependencyResolver, batchingConfiguration);

            Iterable<PluginResource> contextResources = builder.build(new ArrayList<String>(requiredContexts), inclusion.contexts, filter);

            // Check if any context resource has been previously included as single webresources. If so, deliver all
            // contexts as individual webresources without the previously-included webresources.
            boolean isAnyContextAlreadyIncluded = false;
            for (String moduleKey : builder.getAllIncludedResources())
            {
                if (inclusion.webresources.contains(moduleKey))
                {
                    isAnyContextAlreadyIncluded = true;
                }
                else
                {
                    // Add data if not previously included
                    Iterables.addAll(resourcesToInclude.data, pluginResourceLocator.getPluginDataResources(moduleKey));
                }
            }
            if (isAnyContextAlreadyIncluded)
            {
                Iterables.addAll(requiredWebResources, builder.getAllIncludedResources());
            }
            else
            {
                Iterables.addAll(resourcesToInclude.resources, contextResources);
                Iterables.addAll(inclusion.webresources, builder.getAllIncludedResources());
            }
            Iterables.addAll(requiredWebResources, builder.getSkippedResources());
        }

        private Iterable<String> getAllModuleKeysDependencies(Iterable<String> moduleCompleteKeys)
        {
            final Set<String> dependencyModuleCompleteKeys = Sets.newLinkedHashSet();
            for (final String moduleCompleteKey : moduleCompleteKeys)
            {
                final Iterable<String> dependencies = toModuleKeys(dependencyResolver.getDependencies(moduleCompleteKey, batchingConfiguration.isSuperBatchingEnabled()));
                Iterables.addAll(dependencyModuleCompleteKeys, dependencies);
            }
            return dependencyModuleCompleteKeys;
        }

        private void addModuleResources(final Resources resourcesToInclude,
            final Iterable<String> dependencyModuleCompleteKeys, final Set<String> excludeModuleKeys,
            final WebResourceFilter filter)
        {
            for (final String moduleKey : dependencyModuleCompleteKeys)
            {
                if (excludeModuleKeys.contains(moduleKey))
                {
                    // skip this resource if it is already visited
                    continue;
                }

                final List<PluginResource> moduleResources = pluginResourceLocator.getPluginResources(moduleKey);
                for (final PluginResource moduleResource : moduleResources)
                {
                    if (filter.matches(moduleResource.getResourceName()))
                    {
                        resourcesToInclude.resources.add(moduleResource);
                    }
                }
                Iterables.addAll(resourcesToInclude.data, pluginResourceLocator.getPluginDataResources(moduleKey));
            }
        }

        private Iterable<String> toModuleKeys(final Iterable<WebResourceModuleDescriptor> descriptors)
        {
            return transform(descriptors, new TransformDescriptorToKey());
        }
    }
}
