package com.atlassian.plugin.webresource;

import static com.google.common.collect.ImmutableList.copyOf;
import static com.google.common.collect.Iterables.contains;
import static com.google.common.collect.Iterables.transform;
import static com.google.common.collect.Sets.newHashSet;

import com.atlassian.plugin.ModuleDescriptor;
import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.servlet.DownloadableResource;
import com.atlassian.plugin.webresource.url.DefaultUrlBuilder;
import com.google.common.base.Function;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSortedSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Ordering;
import org.apache.commons.codec.binary.Hex;

import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * An intermediary object used for constructing and merging context batches.
 * This is a bean that holds the different resources and parameters that apply
 * to a particular batch.
 * The batch can both include and exclude one or more contexts
 * Resources are expected to be in dependency order, with no duplicates.
 */
class ContextBatch
{
    private static final String UTF8 = "UTF-8";
    private static final String MD5 = "MD5";
    private static final Ordering<ModuleDescriptor<?>> MODULE_KEY_ORDERING = Ordering.natural().onResultOf(new TransformDescriptorToKey());

    private final String key;
    private final List<String> contexts;
    private final Iterable<String> excludedContexts;
    private final Iterable<WebResourceModuleDescriptor> resources;
    private final Iterable<String> resourceKeys;
    private final Set<PluginResourceBatchParams> resourceParams;
    private final boolean removeSuperResources;

    ContextBatch(final String context, final Iterable<WebResourceModuleDescriptor> resources, boolean removeSuperResources)
    {
        this(context, ImmutableList.of(context), resources, ImmutableList.<PluginResourceBatchParams> of(), removeSuperResources);
    }

    /**
     * 
     * @param key
     * @param contexts the ordering of contexts is important since it determines the ordering of resources within a batch (which could be
     * important for badly written Javascripts).
     * @param resources
     * @param resourceParams
     */
    ContextBatch(final String key, final List<String> contexts, final Iterable<WebResourceModuleDescriptor> resources,
                 final Iterable<PluginResourceBatchParams> resourceParams, boolean removeSuperResources)
    {
        this(key, contexts, null, resources, resourceParams, removeSuperResources);
    }
    
    /**
     * 
     * @param key
     * @param contexts the ordering of contexts is important since it determines the ordering of resources within a batch (which could be
     * important for badly written Javascripts).
     * @param excludedContexts the ordering of excluded contexts is not important.
     * @param resources
     * @param resourceParams
     */
    ContextBatch(final String key, final List<String> contexts, Iterable<String> excludedContexts, 
            final Iterable<WebResourceModuleDescriptor> resources, final Iterable<PluginResourceBatchParams> resourceParams,
            boolean removeSuperResources)
    {
        this.key = key;
        this.contexts = copyOf(contexts);
        if (excludedContexts == null)
        {
            this.excludedContexts = Collections.emptyList();
        }
        else
        {
            this.excludedContexts = copyOf(excludedContexts);
        }
        
        this.resourceParams = newHashSet(resourceParams);

        // Note: Ordering is important in producing a consistent hash.
        // But, dependency order is not important when producing the PluginResource,
        // it is only important when we serve the resource. So it is safe to reorder this.
        this.resources = ImmutableSortedSet.copyOf(MODULE_KEY_ORDERING, resources);
        // A convenience object to make searching easier
        this.resourceKeys =  transform(resources, new TransformDescriptorToKey());
        this.removeSuperResources = removeSuperResources;
    }

    public boolean isRemoveSuperResources() {
        return removeSuperResources;
    }

    boolean isResourceIncluded(final String resourceModuleKey)
    {
        return contains(resourceKeys, resourceModuleKey);
    }

    void addResourceType(final PluginResource pluginResource)
    {
        final Map<String, String> parameters = new HashMap<String, String>(PluginResourceLocator.BATCH_PARAMS.length);
        final String type = pluginResource.getType();
        for (final String key : PluginResourceLocator.BATCH_PARAMS)
        {
            if (pluginResource.getParams().get(key) != null)
            {
                parameters.put(key, pluginResource.getParams().get(key));
            }
        }

        resourceParams.add(new PluginResourceBatchParams(type, parameters));
    }

    /**
     * Iterates over the batch parameters for the context ({@link PluginResourceBatchParams}) and
     * creates a {@link ContextBatchPluginResource} for each. 
     * <p/>
     * It should be noted that the created {@link ContextBatchPluginResource} will not actually contain any 
     * {@link DownloadableResource}. The {@link PluginResource} created is primarily representing a
     * URL to be used in constructing a particular resource/download reference.
     *
     * @param pluginAccessor Used to resolve condition and transformation instances
     *  
     * @return ContextBatchPluginResource instances, although containing no {@link DownloadableResource}s.
     */
    Iterable<PluginResource> buildPluginResources(PluginAccessor pluginAccessor, boolean resplitMergedBatches)
    {
        final String versionHash = createHash();

        // add the BatchedWebResourceDescriptors for this ContextBatch to each of the generated ContextBatchPluginResource
        // so that we can return ContextBatches to the client with embedded information about the dependencies contained.
        Set<BatchedWebResourceDescriptor> descriptors = new HashSet<BatchedWebResourceDescriptor>();
        for (WebResourceModuleDescriptor wrmd : this.getResources())
        {
            descriptors.add(new BatchedWebResourceDescriptor(wrmd.getPlugin().getPluginInformation().getVersion(), wrmd.getCompleteKey()));            
        }

        DefaultUrlBuilder urlBuilder = new DefaultUrlBuilder();
        for (WebResourceModuleDescriptor wrmd : this.getResources())
        {
            wrmd.addToUrl(pluginAccessor, urlBuilder);
        }
        
        List<ContextBatchPluginResource> resources = new ArrayList<ContextBatchPluginResource>(resourceParams.size());
        for (PluginResourceBatchParams param :resourceParams)
        {
            ContextBatchPluginResource contextBatchPluginResource = new ContextBatchPluginResource(key, contexts,
                    excludedContexts, versionHash, param.getType(), urlBuilder.allHashes(), param.getParameters(), urlBuilder.toQueryString(), descriptors, removeSuperResources);
            resources.add(contextBatchPluginResource);
        }

        Collections.sort(resources, BatchResourceComparator.INSTANCE);

        return postContextBatchesProcess(resources, resplitMergedBatches);
    }

    private static Iterable<PluginResource> postContextBatchesProcess(List<ContextBatchPluginResource> contextBatchResources, boolean resplitMergedBatches)
    {
        List<PluginResource> result = new LinkedList<PluginResource>();
        if (resplitMergedBatches) {
            for (ContextBatchPluginResource batchResource : contextBatchResources)
            {
                result.addAll(batchResource.splitIntoParts());
            }
        }
        else
        {
            result.addAll(contextBatchResources);
        }
        return result;
    }

    private String createHash()
    {
        try
        {
            MessageDigest md5 = MessageDigest.getInstance(MD5);
            for (WebResourceModuleDescriptor moduleDescriptor : resources)
            {
                String version = moduleDescriptor.getPlugin().getPluginInformation().getVersion();
                md5.update(moduleDescriptor.getCompleteKey().getBytes(UTF8));
                md5.update(version.getBytes(UTF8));
            }

            return new String(Hex.encodeHex(md5.digest()));
        }
        catch (NoSuchAlgorithmException e)
        {
            throw new AssertionError("MD5 hashing algorithm is not available.");
        }
        catch (UnsupportedEncodingException e)
        {
            throw new AssertionError("UTF-8 encoding is not available.");
        }
    }

    String getKey()
    {
        return key;
    }

    List<String> getContexts()
    {
        return contexts;
    }
    
    public Iterable<String> getExcludedContexts()
    {
        return excludedContexts;
    }

    Iterable<WebResourceModuleDescriptor> getResources()
    {
        return resources;
    }
    
    Iterable<String> getResourceKeys()
    {
        return transform(resources, new Function<WebResourceModuleDescriptor,String>() 
        {
            public String apply(WebResourceModuleDescriptor input)
            {
                return input.getCompleteKey();
            }
        });
    }

    Iterable<PluginResourceBatchParams> getResourceParams()
    {
        return Collections.unmodifiableSet(resourceParams);
    }

    @Override
    public boolean equals(Object obj)
    {
        if (obj == null)
            return false;
        
        if (!(obj instanceof ContextBatch))
            return false;
        
        if (obj == this)
            return true;
        
        ContextBatch other = (ContextBatch)obj;
        
        return this.key.equals(other.key);
    }

    @Override
    public int hashCode()
    {
        return 17 * key.hashCode();
    }

    @Override
    public String toString()
    {
        StringBuilder builder = new StringBuilder("ContextBatch[key=");
        builder.append(getKey());
        
        if (contexts != null && Iterables.size(contexts) > 0)
        {
            builder.append(", includedContexts=").append(Iterables.toString(contexts));
        }
        
        if (excludedContexts != null && Iterables.size(excludedContexts) > 0)
        {
            builder.append(", excludedContexts=").append(Iterables.toString(excludedContexts));
        }
        
        if (resourceKeys != null && Iterables.size(resourceKeys) > 0)
        {
            builder.append(", resourceKeys=").append(Iterables.toString(resourceKeys));
        }
        
        builder.append("]");
        
        return builder.toString();
    }
}
