package com.atlassian.plugin.webresource;

import static com.google.common.collect.Iterables.concat;
import static com.google.common.collect.Iterables.transform;

import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Function;
import com.google.common.collect.Iterables;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Performs a calculation on many referenced contexts, and produces an set of intermingled batched-contexts and residual
 * (skipped) resources. Some of the input contexts may have been merged into cross-context batches.
 * The batches are constructed in such a way that no batch is dependent on another.
 * The output batches and resources may be intermingled so as to preserve the input order as much as possible.
 *
 * @since 2.9.0
 */
class ContextBatchBuilder
{
    private static final Logger log = LoggerFactory.getLogger(ContextBatchBuilder.class);

    private final PluginResourceLocator pluginResourceLocator;
    private final ResourceDependencyResolver dependencyResolver;
    private final ResourceBatchingConfiguration batchingConfiguration;

    private final List<String> allIncludedResources = new ArrayList<String>();
    private final Set<String> skippedResources = new HashSet<String>();

    ContextBatchBuilder(final PluginResourceLocator pluginResourceLocator, final ResourceDependencyResolver dependencyResolver, ResourceBatchingConfiguration batchingConfiguration)
    {
        this.pluginResourceLocator = pluginResourceLocator;
        this.dependencyResolver = dependencyResolver;
        this.batchingConfiguration = batchingConfiguration;
    }

    /**
     * This method performs the same function as calling the newer
     * {@link #build(List includedContexts, List excludedContexts)}
     * however it will create merged contexts with a slightly different ordering in the produced URL which of course can
     * lead to a different ordering of resources in the batch. This could cause problems, for instance with Javascripts which
     * have dependencies on each other.
     * <p/>
     * The newer method has a more logical ordering of contexts, <br/>
     * - e.g. the first one requested will always be the first in the URL. <br/>
     * - The next context in the URL will always be the first in the list with an overlapping resource. <br/>
     * Therefore you should prefer the newer {@link #build(List includedContexts, List excludedContexts, WebResourceFilter filter)}
     * and this method will be removed eventually in a future release.
     *
     * @deprecated since 2.12.3. Use {@link #build(java.util.List, java.util.List)} instead.
     */
    Iterable<PluginResource> build(final List<String> includedContexts)
    {
        return build(includedContexts, DefaultWebResourceFilter.INSTANCE);
    }

    /**
     * This method performs the same function as calling the newer 
     * {@link #build(List includedContexts, List excludedContexts, WebResourceFilter filter)}
     * however it will create merged contexts with a slightly different ordering in the produced URL which of course can
     * lead to a different ordering of resources in the batch. This could cause problems, for instance with Javascripts which 
     * have dependencies on each other.
     * <p/>
     * The newer method has a more logical ordering of contexts, <br/>
     * - e.g. the first one requested will always be the first in the URL. <br/>
     * - The next context in the URL will always be the first in the list with an overlapping resource. <br/> 
     * Therefore you should prefer the newer {@link #build(List includedContexts, List excludedContexts, WebResourceFilter filter)}
     * and this method will be removed eventually in a future release.
     * 
     * @deprecated since 2.12. Use {@link #build(java.util.List, java.util.List, WebResourceFilter)} instead.
     */
    Iterable<PluginResource> build(final Iterable<String> includedContexts, final WebResourceFilter filter)
    {
        if (!batchingConfiguration.isContextBatchingEnabled())
        {
             return getUnbatchedResources(includedContexts, null, filter);
        }

        final ContextBatchOperations contextBatchOperations = new ContextBatchOperations(pluginResourceLocator, filter);
        
        // There are three levels to consider here. In order:
        // 1. Type (CSS/JS)
        // 2. Parameters (ieOnly, media, etc)
        // 3. Context
        final List<ContextBatch> batches = new ArrayList<ContextBatch>();

        for (final String context : includedContexts)
        {
            final ContextBatch contextBatch = new ContextBatch(context, dependencyResolver.getDependenciesInContext(context, skippedResources));
            final List<ContextBatch> mergeList = new ArrayList<ContextBatch>();
            for (final WebResourceModuleDescriptor contextResource : contextBatch.getResources())
            {
                // only go deeper if it is not already included
                if (!allIncludedResources.contains(contextResource.getCompleteKey()))
                {
                    for (final PluginResource pluginResource : pluginResourceLocator.getPluginResources(contextResource.getCompleteKey()))
                    {
                        if (filter.matches(pluginResource.getResourceName()))
                        {
                            contextBatch.addResourceType(pluginResource);
                        }
                    }

                    allIncludedResources.add(contextResource.getCompleteKey());
                }
                else
                {
                    // we have an overlapping context, find it.
                    // IMPORTANT: Don't add the overlapping resource to the batch otherwise there'll be duplicates
                    for (final ContextBatch batch : batches)
                    {
                        if (!mergeList.contains(batch) && batch.isResourceIncluded(contextResource.getCompleteKey()))
                        {
                            if (log.isDebugEnabled())
                            {
                                log.debug("Context: {} shares a resource with {}: {}", new String[] { context, batch.getKey(), contextResource.getCompleteKey() });
                            }

                            mergeList.add(batch);
                        }
                    }
                }
            }

            // Merge all the flagged contexts
            if (!mergeList.isEmpty())
            {
                ContextBatch mergedBatch = mergeList.get(0);
                batches.remove(mergedBatch);

                for (int i = 1; i < mergeList.size(); i++)
                {
                    final ContextBatch mergingBatch = mergeList.get(i);
                    mergedBatch = contextBatchOperations.merge(Arrays.asList(mergedBatch, mergingBatch));
                    batches.remove(mergingBatch);
                }

                mergedBatch = contextBatchOperations.merge(Arrays.asList(mergedBatch, contextBatch));
                batches.add(mergedBatch);
            }
            else
            {
                // Otherwise just add a new one
                batches.add(contextBatch);
            }
        }

        // Build the batch resources
        return concat(transform(batches, new Function<ContextBatch, Iterable<PluginResource>>()
        {
            public Iterable<PluginResource> apply(final ContextBatch batch)
            {
                return batch.buildPluginResources();
            }
        }));
    }
        
    Iterable<PluginResource> build(final List<String> includedContexts, final List<String> excludedContexts)
    {
        return build(includedContexts, excludedContexts, DefaultWebResourceFilter.INSTANCE);
    }
    
    Iterable<PluginResource> build(final List<String> includedContexts, final List<String> excludedContexts, final WebResourceFilter filter)
    {
        if (batchingConfiguration.isContextBatchingEnabled())
        {
            return buildBatched(includedContexts, excludedContexts, filter);
        }
        else
        {
            return getUnbatchedResources(includedContexts, excludedContexts, filter);
        }
    }
    
    /**
     * @param includedContexts the ordering of these contexts is important since their placement within the resultant URL determines the order that resources
     * will be included in the batch.
     * @param excludedContexts order of these contexts is not important, they do not affect the position of resources. Instead they cause resources not to
     * be present.
     * @param filter
     * @return
     * @since 2.12
     */
    private Iterable<PluginResource> buildBatched(final List<String> includedContexts, final List<String> excludedContexts, final WebResourceFilter filter)
    {
        Set<String> conditionalIncludedResources = new HashSet<String>();
        WebResourceKeysToContextBatches includedBatches = WebResourceKeysToContextBatches.create(includedContexts, dependencyResolver, pluginResourceLocator, filter, conditionalIncludedResources);
        WebResourceKeysToContextBatches excludedBatches = null;
        if (excludedContexts != null && !Iterables.isEmpty(excludedContexts)) 
        {
            excludedBatches = WebResourceKeysToContextBatches.create(excludedContexts, dependencyResolver, pluginResourceLocator, filter, new HashSet<String>());
        }
        
        skippedResources.addAll(includedBatches.getSkippedResources());
        
        // There are three levels to consider here. In order:
        // 1. Type (CSS/JS)
        // 2. Parameters (ieOnly, media, etc)
        // 3. Context
        final List<ContextBatch> batches = new ArrayList<ContextBatch>();

        // This working list will be reduced as each context is handled.
        final List<ContextBatch> batchesToProcess = new ArrayList<ContextBatch>(includedBatches.getContextBatches());
        
        final ContextBatchOperations contextBatchOperations = new ContextBatchOperations(pluginResourceLocator, filter);
        
        while (!batchesToProcess.isEmpty())
        {
            ContextBatch contextBatch = batchesToProcess.remove(0);
            Set<ContextBatch> alreadyProcessedBatches = new HashSet<ContextBatch>();
            alreadyProcessedBatches.add(contextBatch);

            Iterator<WebResourceModuleDescriptor> resourceIterator = contextBatch.getResources().iterator();
            while (resourceIterator.hasNext())
            {
                WebResourceModuleDescriptor contextResource = resourceIterator.next();
                String resourceKey = contextResource.getCompleteKey();
                // check for an overlap with the other batches (take into account only the batches not yet processed).
                List<ContextBatch> additionalContexts = includedBatches.getAdditionalContextsForResourceKey(resourceKey, alreadyProcessedBatches);

                if (CollectionUtils.isNotEmpty(additionalContexts))
                {
                    if (log.isDebugEnabled())
                    {
                        for (ContextBatch additional : additionalContexts)
                        {
                            log.debug("Context: {} shares a resource with {}: {}", new String[] { contextBatch.getKey(), additional.getKey(), contextResource.getCompleteKey() });
                        }
                    }

                    List<ContextBatch> contextsToMerge = new ArrayList<ContextBatch>(1 + additionalContexts.size());
                    contextsToMerge.add(contextBatch);
                    contextsToMerge.addAll(additionalContexts);
                    contextBatch = contextBatchOperations.merge(contextsToMerge);
                
                    // remove the merged batches from those to be processed
                    batchesToProcess.removeAll(additionalContexts);
                    alreadyProcessedBatches.addAll(additionalContexts);

                    // As a new overlapping context is merged, restart the resource iterator so we can check for new resources 
                    // that may have been added via the merge.
                    resourceIterator = contextBatch.getResources().iterator();
                }
            }
            
            // We separate the search for excluded batches since we want to perform the subtraction
            // after all the merging has been done. If you do a subtraction then a merge you cannot
            // ensure (with ContextBatch as it is currently implemented) that the merge will not
            // bring back in previously excluded resources.
            if (excludedBatches != null)
            {                
                resourceIterator = contextBatch.getResources().iterator();
                while (resourceIterator.hasNext())
                {
                    WebResourceModuleDescriptor contextResource = resourceIterator.next();
                    String resourceKey = contextResource.getCompleteKey();
                    
                    List<ContextBatch> excludeContexts = excludedBatches.getContextsForResourceKey(resourceKey);
                    if (!excludeContexts.isEmpty())
                    {
                        contextBatch = contextBatchOperations.subtract(contextBatch, excludeContexts);
                    }
                }
                
                skippedResources.removeAll(excludedBatches.getSkippedResources());
            }
            
            // check that we still have resources in this batch - if not, the batch is not required.
            if (excludedBatches == null || Iterables.size(contextBatch.getResources()) != 0)
            {
                Iterables.addAll(allIncludedResources, contextBatch.getResourceKeys());
                batches.add(contextBatch);                
            }
            else if (log.isDebugEnabled())
            {
                log.debug("The context batch {} contains no resources so will be dropped.", contextBatch.getKey());
            }                
        }
        
        // Build the batch resources
        return concat(transform(batches, new Function<ContextBatch, Iterable<PluginResource>>()
        {
            public Iterable<PluginResource> apply(final ContextBatch batch)
            {
                return batch.buildPluginResources();
            }
        }));
    }

    // If context batching is not enabled, then just add all the resources that would have been added in the context anyway.
    private Iterable<PluginResource> getUnbatchedResources(final Iterable<String> includedContexts, final Iterable<String> excludedContexts, final WebResourceFilter filter)
    {        
        Set<String> excludedResourceKeys = new HashSet<String>();
        Set<String> excludedSkippedResources = new HashSet<String>(); // the resources that an excluded batch will not contain
        
        if (excludedContexts != null && Iterables.size(excludedContexts) > 0)
        {
            for (final String context : excludedContexts)
            {
                Iterable<WebResourceModuleDescriptor> contextResources = dependencyResolver.getDependenciesInContext(context, excludedSkippedResources);
                for (final WebResourceModuleDescriptor contextResource : contextResources)
                {
                    excludedResourceKeys.add(contextResource.getCompleteKey());
                }
            }            
        }
        
        LinkedHashSet<PluginResource> includedResources = new LinkedHashSet<PluginResource>();
        Set<String> includedSkippedResources = new HashSet<String>(); // the resources that an included batch will not contain
        
        for (final String context : includedContexts)
        {
            Iterable<WebResourceModuleDescriptor> contextResources = dependencyResolver.getDependenciesInContext(context, includedSkippedResources);

            for (final WebResourceModuleDescriptor contextResource : contextResources)
            {
                String completeKey = contextResource.getCompleteKey();
                if (!excludedResourceKeys.contains(completeKey) && !allIncludedResources.contains(completeKey))
                {
                    final List<PluginResource> moduleResources = pluginResourceLocator.getPluginResources(contextResource.getCompleteKey());
                    for (final PluginResource moduleResource : moduleResources)
                    {
                        if (filter.matches(moduleResource.getResourceName()))
                        {
                            includedResources.add(moduleResource);
                        }
                    }

                    allIncludedResources.add(contextResource.getCompleteKey());
                }                
            }
        }
        
        includedSkippedResources.removeAll(excludedSkippedResources);
        skippedResources.addAll(includedSkippedResources);

        return includedResources;
    }

    Iterable<String> getAllIncludedResources()
    {
        return allIncludedResources;
    }

    Iterable<String> getSkippedResources()
    {
        return skippedResources;
    }
    
    
    private static class WebResourceKeysToContextBatches
    {
        /**
         * Create a Map of {@link WebResourceModuleDescriptor} key to the contexts that includes them. If a 
         * {@link WebResourceModuleDescriptor} exists in multiple contexts then each context will be 
         * referenced. The ContextBatches created at this point are pure - they do not take into account
         * overlaps or exclusions, they simply contain all the resources for their context name.
         * 
         * @param contexts the contexts to create a mapping for
         * @param dependencyResolver used to construct the identified contexts
         * @param pluginResourceLocator used to find the individual resources for a WebResourceModuleDescriptor before filtering them.
         * @param filter the filter selecting the resources to be included in the batch
         * @param conditionalResources conditional resources cannot be included in a batch so will be added to this list instead.
         * @return a WebResourceToContextsMap containing the required mapping.
         */        
        static WebResourceKeysToContextBatches create(final List<String> contexts, final ResourceDependencyResolver dependencyResolver, PluginResourceLocator pluginResourceLocator, final WebResourceFilter filter, Set<String> conditionalResources)
        {
            final Map<String, List<ContextBatch>> resourceKeyToContext = new HashMap<String,List<ContextBatch>>();
            final List<ContextBatch> batches = new ArrayList<ContextBatch>();
            final Set<String> skippedResources = new HashSet<String>();
            
            for (String context : contexts)
            {
                Iterable<WebResourceModuleDescriptor> dependencies = dependencyResolver.getDependenciesInContext(context, skippedResources);
                
                ContextBatch batch = new ContextBatch(context, dependencies);
                for (WebResourceModuleDescriptor moduleDescriptor : dependencies)
                {
                    String key = moduleDescriptor.getCompleteKey();
                    boolean matchedPluginResource = false;

                    for (final PluginResource pluginResource : pluginResourceLocator.getPluginResources(moduleDescriptor.getCompleteKey()))
                    {
                        if (filter.matches(pluginResource.getResourceName()))
                        {
                            batch.addResourceType(pluginResource);
                            matchedPluginResource = true;
                        }
                    }
                    
                    if (matchedPluginResource)
                    {
                        if (!resourceKeyToContext.containsKey(key))
                        {
                            resourceKeyToContext.put(key, new ArrayList<ContextBatch>());
                        }
                        
                        resourceKeyToContext.get(key).add(batch);
                        
                        if (!batches.contains(batch))
                        {
                            batches.add(batch);
                        }
                    }
                }
                
            }
            
            return new WebResourceKeysToContextBatches(resourceKeyToContext, batches, skippedResources);
        }
        
        private final Map<String, List<ContextBatch>> resourceToContextBatches; 
        private final List<ContextBatch> knownBatches;
        private final Set<String> skippedResources;        
        
        private WebResourceKeysToContextBatches(Map<String, List<ContextBatch>> resourceKeyToContext, List<ContextBatch> allBatches, Set<String> skippedResources)
        {
            this.resourceToContextBatches = resourceKeyToContext;
            this.knownBatches = allBatches;
            this.skippedResources = skippedResources;
        }
        
        /**
         * @param key the resource key to find contexts for
         * @return all contexts the specified resource is included in. An empty List is returned if none.
         */
        List<ContextBatch> getContextsForResourceKey(String key)
        {
            return getAdditionalContextsForResourceKey(key, null);
        }        
        
        /**
         * @param key the resource key to be mapped to contexts
         * @param knownContexts the contexts we already know about (may be null)
         * @return a List of any additional contexts that the identified resource key can be found in. If there
         * are no additional contexts then an empty List is returned.
         */
        List<ContextBatch> getAdditionalContextsForResourceKey(String key, Collection<ContextBatch> knownContexts)
        {
            List<ContextBatch> allContexts = resourceToContextBatches.get(key);
            if (CollectionUtils.isEmpty(allContexts))
            {
                return Collections.emptyList();
            }
            
            LinkedHashSet<ContextBatch> contexts = new LinkedHashSet<ContextBatch>(allContexts);
            if (CollectionUtils.isNotEmpty(knownContexts))
            {
                contexts.removeAll(knownContexts);
            }

            return new ArrayList<ContextBatch>(contexts);
        }
        
        /**
         * @return all the ContextBatches referenced in this class.
         */
        List<ContextBatch> getContextBatches()
        {
            return new ArrayList<ContextBatch>(knownBatches);
        }

        /**
         * @return the conditional resources that could not be mapped
         */
        public Set<String> getSkippedResources()
        {
            return skippedResources;
        }
    }
}
