package com.atlassian.plugin.servlet.cache.model;

import javax.annotation.Nonnull;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpServletResponseWrapper;
import java.io.*;
import java.util.Optional;

import static com.atlassian.plugin.servlet.util.function.FailableSupplier.wrapper;
import static java.util.Objects.nonNull;
import static java.util.Optional.ofNullable;

/**
 * <p>Represents a {@link HttpServletResponseWrapper} responsible for exposing the content of a {@link HttpServletResponse} as {@link ETagToken}.</p>
 *
 * @since 4.1.18
 */
public class CacheableResponse extends HttpServletResponseWrapper implements Closeable {

    private CacheableResponseStream outputStream;
    private PrintWriter writer;

    public CacheableResponse(@Nonnull final HttpServletResponse response) {
        super(response);
    }

    @Override
    public final void close() throws IOException {
        if (nonNull(writer)) {
            writer.close();
        }
        if (nonNull(outputStream)) {
            outputStream.close();
        }
    }

    @Override
    public final void flushBuffer() throws IOException {
        if (nonNull(writer)) {
            writer.flush();
        }
        if (nonNull(outputStream)) {
            outputStream.flush();
        }
    }

    public final Optional<byte[]> getContentBody() {
        return ofNullable(outputStream)
                .map(CacheableResponseStream::getCopy);
    }

    @Override
    public final ServletOutputStream getOutputStream() {
        outputStream = ofNullable(outputStream)
                .orElseGet(CacheableResponseStream::new);
        return outputStream;
    }

    @Override
    public final PrintWriter getWriter() {
        writer = ofNullable(writer)
                .orElseGet(wrapper(() -> {
                    final String characterEncoding = getResponse().getCharacterEncoding();
                    final OutputStreamWriter outputStreamWriter = new OutputStreamWriter(getOutputStream(), characterEncoding);
                    return new PrintWriter(outputStreamWriter, true);
                }));
        return writer;
    }

    /**
     * Get the representation of the current response body as {@link ETagToken}.
     * @return A possible representation of the current response body if the body is not null.
     */
    public Optional<ETagToken> toETagToken() {
        return getContentBody()
                .map(ETagToken::new);
    }
}
