package com.atlassian.oauth2.scopes.api;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Supplier;

public interface ScopesRequestCache {

    /**
     * Applies scopes to Request cache. The scopes will be removed by calling close() method on the Closeable.
     * @param scope the scopes applied during the action
     * @param applicationNameSupplier attempts to supply the application name for the request
     * @return Closeable to remove the scopes
     * @since 2.1
     */
    Closeable withScopes(final Scope scope, final Supplier<Optional<String>> applicationNameSupplier);

    /**
     * Checks if the cache contains only the passed scope
     * @param scope the scope to check if it exists alone in the cache
     * @return true if it exists alone, false otherwise
     */
    boolean containsOnlyThisScope(Scope scope);

    /**
     * Using the scope cache, checks if the request has the specified permission
     * @param permission the permission to check
     * @return true if the scope cache is empty, or the request has permission. Otherwise false.
     */
    boolean hasPermission(Permission permission);

    /**
     * Gets the application name for the OAuth 2 request if this is one
     * Otherwise returns an empty Optional
     * @return an Optional with the Application Name or an empty Optional
     */
    @Nonnull
    Optional<String> getApplicationNameForRequest();

    /**
     * Gets the current set of permissions for the request
     * @return the set of Permissions for the current request or an empty set if none present
     */
    @Nonnull
    Set<Permission> getPermissionsForRequest();

    /**
     * Gets a copy of the stored request cache
     * @return
     */
    RequestCache getRequestCache();

    /**
     * Allows the request cache to be manually set.
     * Setting it to null is the same as clearing it
     * @param requestCache the request cache to store
     */
    void setRequestCache(@Nullable RequestCache requestCache);

    /**
     * Removes the scopes request cache.
     */
    void clearRequestCache();

    /**
     * The state of the ScopesRequestCache
     */
    class RequestCache {
        private final Set<Permission> permissions;
        private final Supplier<Optional<String>> applicationNameSupplier;

        public RequestCache() {
            this.permissions = Collections.emptySet();
            this.applicationNameSupplier = Optional::empty;
        }

        public RequestCache(final Collection<Permission> permissions, final Supplier<Optional<String>> applicationNameSupplier) {
            this.permissions = new HashSet<Permission>(permissions) {
                @Override
                public boolean contains(final Object permission) {
                    return this.stream()
                            .anyMatch(setPermission -> setPermission.equals(permission));
                }
            };
            this.applicationNameSupplier = applicationNameSupplier != null
                    ? applicationNameSupplier
                    : Optional::empty;
        }

        public Set<Permission> getPermissions() {
            return permissions;
        }

        public Supplier<Optional<String>> getApplicationNameSupplier() {
            return applicationNameSupplier;
        }

        public RequestCache copy() {
            return new RequestCache(permissions, applicationNameSupplier);
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;
            final RequestCache that = (RequestCache) o;
            return Objects.equals(permissions, that.permissions) && Objects.equals(applicationNameSupplier, that.applicationNameSupplier);
        }

        @Override
        public int hashCode() {
            return Objects.hash(permissions, applicationNameSupplier);
        }
    }
}
