package com.atlassian.oauth2.provider.api.token;

import com.atlassian.oauth2.provider.api.token.access.AccessToken;
import com.atlassian.oauth2.provider.api.token.refresh.RefreshToken;
import com.atlassian.oauth2.scopes.api.Scope;
import com.atlassian.sal.api.user.UserKey;

import javax.annotation.Nonnull;
import java.time.Duration;
import java.util.List;
import java.util.Optional;

public interface TokenService {

    /**
     * Creates an Access Token
     *
     * @param clientId client id for the token
     * @param userKey userKey for token
     * @param authorizationCode authorization code used to redeem token
     * @param authorizationDate date when authorization was given
     * @param scope scope for access token
     * @return created access token
     */
    @Nonnull
    AccessToken createAccessToken(String clientId,
                                  String userKey,
                                  String authorizationCode,
                                  long authorizationDate,
                                  Scope scope);

    /**
     * Creates a Refresh Token
     *
     * @param clientId client id for the token
     * @param userKey userKey for token
     * @param authorizationDate date when authorization was given
     * @param accessTokenId associated with refresh token
     * @param scope scope for refresh token
     * @param refreshCount the number of refreshes
     * @param authorizationCode authorization code used to redeem token
     * @return created refresh token
     */
    @Nonnull
    RefreshToken createRefreshToken(String clientId,
                                    String userKey,
                                    long authorizationDate,
                                    String accessTokenId,
                                    Scope scope,
                                    String authorizationCode,
                                    int refreshCount);

    /**
     * Updates last accessed time for access token
     *
     * @param tokenId token to update
     */
    void updateAccessTokenLastAccessed(String tokenId);

    /**
     * Tries to find a {@link AccessToken} using the raw token
     *
     * @param tokenId raw token to search database with
     * @return an optional token
     */
    @Nonnull
    Optional<AccessToken> findByAccessTokenId(String tokenId);

    /**
     * Tries to find a {@link RefreshToken} using the raw token
     *
     * @param tokenId raw token to search database with
     * @return an optional token
     */
    @Nonnull
    Optional<RefreshToken> findByRefreshTokenId(String tokenId);

    /**
     * Removes stored token using raw token
     *
     * @param tokenId id of the token to remove
     * @return removed token or empty
     */
    @Nonnull
    Optional<AccessToken> removeAccessTokenById(String tokenId);

    /**
     * Removes stored tokens using the provided authorization code
     *
     * @param authorizationCode authorization code to search for
     * @return removed tokens or empty
     */
    @Nonnull
    List<RefreshToken> removeTokensByAuthorizationCode(String authorizationCode);

    /**
     * Removes all tokens for client id
     *
     * @param clientId client id whose tokens are to be removed
     */
    void removeByClientId(String clientId);

    /**
     * Removes all tokens for user key
     *
     * @param userKey user key for whose tokens are to be removed
     * @return removed tokens or empty
     */
    @Nonnull
    List<AccessToken> removeByUserKey(String userKey);

    /**
     * Removes all expired access tokens
     *
     * @param expirationPeriod the period of time after which we remove access tokens
     */
    void removeExpiredAccessTokens(@Nonnull Duration expirationPeriod);

    /**
     * Removes all expired refresh tokens
     *
     * @param expirationPeriod the period of time after which we remove refresh tokens
     */
    void removeExpiredRefreshTokens(@Nonnull Duration expirationPeriod);

    /**
     * Finds user keys of tokens for a client id
     *
     * @param clientId client id with tokens to search for
     * @return found userKeys as strings or an empty list
     */
    @Nonnull
    List<String> findUserKeysByClientId(String clientId);

    /**
     * Finds refresh tokens for a client id
     *
     * @param clientId client id with tokens to search for
     * @return found refreshTokens or an empty list
     */
    @Nonnull
    List<RefreshToken> findRefreshTokensForClientId(String clientId);

    /**
     * Finds access tokens for userKey
     * @param userKey user key to search with
     * @return found tokens or an empty list
     */
    List<AccessToken> findAccessTokensByUserKey(@Nonnull UserKey userKey);

    /**
     * Finds refresh tokens for userKey
     * @param userKey user key to search with
     * @return found tokens or an empty list
     */
    List<RefreshToken> findRefreshTokensByUserKey(@Nonnull UserKey userKey);

    /**
     * Removes token associated with refresh token
     * @return revoked access token
     */
    Optional<AccessToken> removeAccessTokenAssociatedWith(String refreshTokenId);

    /**
     * Removes token associated with access token
     * @return revoked refresh token
     */
    Optional<RefreshToken> removeRefreshTokenAssociatedWith(String accessTokenId);

    /**
     * Checks if the provided access token is stored against the provided client ID
     *
     * @param clientId id of the client
     * @param accessTokenId id of the access token to validate
     * @return true if the refresh token is valid
     */
    boolean isAccessTokenValid(@Nonnull String clientId, @Nonnull String accessTokenId);

    /**
     * Checks if the provided refresh token is stored against the provided client ID
     *
     * @param clientId id of the client
     * @param refreshTokenId id of the refresh token to validate
     * @return true if the refresh token is valid
     */
    boolean isRefreshTokenValid(@Nonnull String clientId, @Nonnull String refreshTokenId);

    /**
     * Removes refresh token
     *
     * @param refreshTokenId - id of the refresh token to remove
     * @return removed {@link RefreshToken} or empty
     */
    Optional<RefreshToken> removeRefreshToken(String refreshTokenId);

    /**
     * Checks if the provided authorization code is held against
     *
     * an existing token
     * @param authorizationCode code to search for
     * @return true if the authorization code is found
     */
    boolean isCodeRedeemed(String authorizationCode);

    /**
     * Removes tokens for tokenId, treating it as an access or refresh token and removing either
     * @param tokenId access or refresh token id
     * @return true if tokens removed, false if no tokens removed
     */
    boolean removeTokensById(@Nonnull String tokenId);
}
