package com.atlassian.oauth2.provider.api.client;

import com.atlassian.oauth2.scopes.api.Scope;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.List;
import java.util.Optional;

/**
 * Service for managing OAuth2 clients (from the provider's perspective).
 *
 * When imported via OSGi, System Admin permission
 * checks are enforced on all method calls.
 */
public interface ClientService {

    /**
     * Creates a {@link Client}
     * @param name the name of the client
     * @param scope optional scope for client. If not specified then this will be saved as ALL
     * @param redirectUris the redirect uris of the client
     * @return the newly created entity
     */
    @Nonnull
    Client create(@Nonnull String name,
                  Scope scope,
                  @Nonnull List<String> redirectUris);

    /**
     * Updates an existing {@link Client}
     * @param id the id of the entity to update
     * @param name the new name. Null means no update will occur
     * @param redirectUris the new redirectUri name
     */
    Optional<Client> updateClient(@Nonnull String id,
                                  String name,
                                  String scope,
                                  @Nonnull List<String> redirectUris);

    Optional<Client> resetClientSecret(@Nonnull String clientId);

    /**
     * Gets an existing {@link Client}
     * @param id of the entity to return
     * @return The {@link Client} matching the id or null
     */
    Optional<Client> getById(@Nonnull String id);

    /**
     * Gets an existing {@link Client}
     * @param clientId of the entity to return
     * @return The {@link Client} matching the id or null
     */
    Optional<Client> getByClientId(@Nonnull String clientId);

    /**
     * Gets an existing redirect uri
     * @param clientId matching the entity to return
     * @return The redirect uri matching the id or empty
     */
    List<String> findRedirectUrisByClientId(@Nonnull String clientId);

    /**
     * Gets all Client
     * @return the clients stored in the system
     */
    List<Client> list();

    /**
     * Deletes a {@link Client}
     * @param id of the entity to delete
     */
    Optional<Client> removeById(@Nonnull String id);

    /**
     * Checks if client name is unique
     * @param clientId id of the client
     * @param clientName name of the client
     * @return true if client name is unique
     */
    boolean isClientNameUnique(@Nullable String clientId, @Nonnull String clientName);

    /**
     * Checks if the provided client secret is a match for the client secret stored against
     * the client ID provided
     * @param clientId id of the client
     * @param clientSecret secret of the client
     * @return true if provided client secret is a match
     */
    boolean isClientSecretValid(@Nonnull String clientId, @Nonnull String clientSecret);
}
