package com.atlassian.oauth2.client.api.storage.config;

import com.atlassian.oauth2.client.api.ClientConfiguration;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

public class ClientConfigurationEntity implements ClientConfiguration {

    private final String id;
    private final String name;
    private final ProviderType providerType;
    private final String description;
    private final String clientId;
    private final String clientSecret;
    private final String authorizationEndpoint;
    private final String tokenEndpoint;
    private final List<String> scopes;

    private ClientConfigurationEntity(final Builder builder) {
        this.id = builder.id;
        this.name = Objects.requireNonNull(builder.name, "Name cannot be null");
        this.description = builder.description;
        this.providerType = Objects.requireNonNull(builder.providerType, "Provider type cannot be null");
        this.clientId = Objects.requireNonNull(builder.clientId, "Client ID cannot be null");
        this.clientSecret = Objects.requireNonNull(builder.clientSecret, "Client secret cannot be null");
        this.authorizationEndpoint = Objects.requireNonNull(builder.authorizationEndpoint, "Authorization endpoint cannot be null");
        this.tokenEndpoint = Objects.requireNonNull(builder.tokenEndpoint, "Token endpoint cannot be null");
        Preconditions.checkArgument(builder.scopes != null && !builder.scopes.isEmpty(), "At least one scope has to be provided");
        this.scopes = ImmutableList.copyOf(builder.scopes);
    }

    @Nullable
    public String getId() {
        return id;
    }

    @Nonnull
    public String getName() {
        return name;
    }

    @Nullable
    public String getDescription() {
        return description;
    }

    @Override
    @Nonnull
    public ProviderType getProviderType() {
        return providerType;
    }

    @Override
    @Nonnull
    public String getClientId() {
        return clientId;
    }

    @Override
    @Nonnull
    public String getClientSecret() {
        return clientSecret;
    }

    @Override
    @Nonnull
    public String getAuthorizationEndpoint() {
        return authorizationEndpoint;
    }

    @Override
    @Nonnull
    public String getTokenEndpoint() {
        return tokenEndpoint;
    }

    @Override
    @Nonnull
    public List<String> getScopes() {
        return scopes;
    }

    @Nonnull
    public Builder toBuilder() {
        return builder(this);
    }

    @Nonnull
    public static Builder builder() {
        return new Builder();
    }

    @Nonnull
    public static Builder builder(@Nonnull final ClientConfiguration data) {
        return new Builder(data);
    }

    @Nonnull
    public static Builder builder(@Nonnull final ClientConfigurationEntity data) {
        return new Builder(data);
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        final ClientConfigurationEntity that = (ClientConfigurationEntity) o;

        return Objects.equals(this.getId(), that.getId())
                && Objects.equals(this.getName(), that.getName())
                && Objects.equals(this.getDescription(), that.getDescription())
                && Objects.equals(this.getProviderType(), that.getProviderType())
                && Objects.equals(this.getClientId(), that.getClientId())
                && Objects.equals(this.getClientSecret(), that.getClientSecret())
                && Objects.equals(this.getAuthorizationEndpoint(), that.getAuthorizationEndpoint())
                && Objects.equals(this.getTokenEndpoint(), that.getTokenEndpoint())
                && Objects.equals(this.getScopes(), that.getScopes());
    }

    @Override
    public int hashCode() {
        return Objects.hash(
                getId(), getName(), getDescription(), getProviderType(),
                getClientId(), getClientSecret(), getAuthorizationEndpoint(), getTokenEndpoint(), getScopes()
        );
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("id", getId())
                .add("name", getName())
                .add("description", getDescription())
                .add("providerType", getProviderType())
                .add("clientId", getClientId())
                .add("clientSecret", "*****")
                .add("authorizationEndpoint", getAuthorizationEndpoint())
                .add("tokenEndpoint", getTokenEndpoint())
                .add("scopes", getScopes())
                .toString();
    }

    public static final class Builder {

        private String id;
        private String name;
        private String description;
        private ProviderType providerType;
        private String clientId;
        private String clientSecret;
        private String authorizationEndpoint;
        private String tokenEndpoint;
        private List<String> scopes = new ArrayList<>();

        private Builder() {
        }

        private Builder(@Nonnull final ClientConfiguration initialData) {
            this.clientId = initialData.getClientId();
            this.clientSecret = initialData.getClientSecret();
            this.authorizationEndpoint = initialData.getAuthorizationEndpoint();
            this.tokenEndpoint = initialData.getTokenEndpoint();
            this.scopes = new ArrayList<>(initialData.getScopes());
            this.providerType = initialData.getProviderType();
        }

        private Builder(@Nonnull final ClientConfigurationEntity initialData) {
            this((ClientConfiguration) initialData);
            this.id = initialData.getId();
            this.description = initialData.getDescription();
            this.name = initialData.getName();
        }

        public Builder id(@Nullable final String id) {
            this.id = id;
            return this;
        }

        public Builder name(@Nonnull final String name) {
            this.name = name;
            return this;
        }

        public Builder description(@Nullable final String description) {
            this.description = description;
            return this;
        }

        public Builder providerType(@Nonnull final ProviderType providerType) {
            this.providerType = providerType;
            return this;
        }

        public Builder clientId(@Nonnull final String clientId) {
            this.clientId = clientId;
            return this;
        }

        public Builder clientSecret(@Nonnull final String clientSecret) {
            this.clientSecret = clientSecret;
            return this;
        }

        public Builder authorizationEndpoint(@Nonnull final String authorizationEndpoint) {
            this.authorizationEndpoint = authorizationEndpoint;
            return this;
        }

        public Builder tokenEndpoint(@Nonnull final String tokenEndpoint) {
            this.tokenEndpoint = tokenEndpoint;
            return this;
        }

        public Builder scopes(@Nonnull final List<String> scopes) {
            this.scopes = scopes;
            return this;
        }

        public Builder addScope(@Nonnull final String scope) {
            this.scopes.add(scope);
            return this;
        }

        public ClientConfigurationEntity build() {
            return new ClientConfigurationEntity(this);
        }
    }
}
