package com.atlassian.integrationtesting.ui;

import com.atlassian.integrationtesting.runner.ApplicationLogDelimiter;
import com.atlassian.integrationtesting.runner.CompositeTestRunner;
import com.atlassian.integrationtesting.runner.TestGroupRunner;
import com.atlassian.integrationtesting.runner.restore.RestoreFromBackup;
import com.atlassian.integrationtesting.ui.UiRunListener.OutputDirectory;
import com.atlassian.sal.api.ApplicationProperties;

import com.google.common.base.Function;
import com.google.common.collect.ImmutableList;
import com.google.inject.AbstractModule;
import com.google.inject.ConfigurationException;
import com.google.inject.Guice;
import com.google.inject.Injector;
import com.google.inject.Module;
import com.google.inject.Scopes;
import com.google.inject.spi.Message;

import org.junit.runners.model.InitializationError;

import static org.apache.commons.lang.StringUtils.isBlank;

/**
 * <p>A test runner which will inject a {@link UiTester} before a running tests in a test class.  It does this by
 * looking for static fields whose type is {@code UiTester} or a type implementing the {@code UiTester} interface. 
 * {@code UiTester}s are created using the {@link UiTesterFactory}, which users of this class are expected to provide.
 * When all the tests in a test class are finished, the {@link UiTester#destroy()} method is called for cleanup.</p>
 *   
 * <p>In addition to providing creation, injection and cleanup of {@code UiTester}s, this runner also adds instances
 * of {@link UiRunListener}, so that test failures will cause the contents of the current page the {@code UiTester} was
 * working with to be dumped to a log file.</p>
 * 
 * <p>To top off all of that, the predicates from {@code TestGroupRunner} are also mixed in.</p>
 */
public abstract class UiTestRunner extends CompositeTestRunner
{
    public UiTestRunner(Class<?> klass, Module... modules) throws InitializationError
    {
        super(klass, compose(modules));
    }

    /**
     * Creates a {@code Composer} using the given modules.  The modules should be configured with bindings for an
     * {@link ApplicationProperties} and {@link UiTester} singleton instance.  If it hasn't been properly configured,
     * a configuration exception will be thrown.
     * 
     * @param modules Modules providing a binding for {@code ApplicationProperties} and {@link UiTester}
     * @return composer composite that has functions from {@link TestGroupRunner} and {@link ApplicationLogDelimiter},
     * supports static injection of the test class, and adds a {@link UiRunListener} before tests are run.  
     */
    public static Composer compose(Module... modules)
    {
        Injector injector = Guice.createInjector(ImmutableList.<Module>builder().add(modules).add(new RunningTestGroupModule()).build());
        return compose(injector);
    }

    /**
     * Creates a {@code Composer} using the given {@coe Injector} as a parent.  The injector should be able to provide
     * singleton instances of {@link ApplicationProperties}, {@link UiTester} and a binding for {@link File} fields and
     * parameters with the {@link OutputDirectory} annotation.  If it hasn't been properly configured, a configuration
     * exception will be thrown.
     * 
     * @param injector Injector providing a binding for {@code ApplicationProperties}, {@link UiTester} and {@link OutputDirectory}
     * @return composer composite that has functions from {@link TestGroupRunner} and {@link ApplicationLogDelimiter},
     * supports static injection of the test class, and adds a {@link UiRunListener} before tests are run.  
     */
    public static Composer compose(Injector injector)
    {
        Injector child = injector.createChildInjector(new UiTestRunnerModule());
        return compose().
            from(TestGroupRunner.compose()).
            from(ApplicationLogDelimiter.compose(child)).
            beforeTestClass(new InjectStatics(child)).
            beforeTestClass(new AddListener(child)).
            afterTestClass(new RemoveListener(child)).
            afterTestClass(new DestroyUiTester(child)).
            from(RestoreFromBackup.compose(child));
    }

    private static final class UiTestRunnerModule extends AbstractModule
    {
        @Override
        protected void configure()
        {
            bind(UiRunListener.class).in(Scopes.SINGLETON);
        }
    }
    
    private static final class RunningTestGroupModule extends AbstractModule
    {
        @Override
        protected void configure()
        {
            if (isBlank(TestGroupRunner.getRunningTestGroup()))
            {
                throw new ConfigurationException(ImmutableList.of(new Message("No testGroup configured - can't figure out which UiTester to use without a test group in the form {application}-v{version}")));
            }
            bind(String.class).annotatedWith(RunningTestGroup.class).toInstance(TestGroupRunner.getRunningTestGroup());
        }
    }
    
    private static final class InjectStatics implements Function<BeforeTestClass, Void>
    {
        private final Injector injector;

        public InjectStatics(Injector injector)
        {
            this.injector = injector;
        }

        public Void apply(BeforeTestClass test)
        {
            injector.createChildInjector(new StaticInjectionModule(test.testClass.getJavaClass()));
            return null;
        }
        
        private static final class StaticInjectionModule extends AbstractModule
        {
            private final Class<?> testClass;

            public StaticInjectionModule(Class<?> testClass)
            {
                this.testClass = testClass;
            }

            @Override
            protected void configure()
            {
                requestStaticInjection(testClass);
            }
        }
    }
    
    private static final class AddListener implements Function<BeforeTestClass, Void>
    {
        private final Injector injector;

        public AddListener(Injector injector)
        {
            this.injector = injector;
        }

        public Void apply(BeforeTestClass test)
        {
            test.notifier.addListener(injector.getInstance(UiRunListener.class));
            return null;
        }
    }
    
    private static final class RemoveListener implements Function<AfterTestClass, Void>
    {
        private final Injector injector;

        public RemoveListener(Injector injector)
        {
            this.injector = injector;
        }

        public Void apply(AfterTestClass test)
        {
            test.notifier.removeListener(injector.getInstance(UiRunListener.class));
            return null;
        }
    }
    
    private static final class DestroyUiTester implements Function<AfterTestClass, Void>
    {
        private final Injector injector;

        public DestroyUiTester(Injector injector)
        {
            this.injector = injector;
        }

        public Void apply(AfterTestClass from)
        {
            injector.getInstance(UiTester.class).destroy();
            return null;
        }
    }
}
