package com.atlassian.integrationtesting.ui;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

import com.google.inject.Inject;

import org.junit.runner.notification.Failure;
import org.junit.runner.notification.RunListener;

import static org.apache.commons.io.IOUtils.closeQuietly;

/**
 * A {@code RunListener} for HtmlUnit UI tests.  It listens for test failures and when one occurs dumps the content
 * of the current page that HtmlUnit is on.
 */
public class UiRunListener extends RunListener
{
    @Retention(RetentionPolicy.RUNTIME)
    @Target({ElementType.METHOD, ElementType.PARAMETER, ElementType.FIELD})
    public @interface OutputDirectory{}

    private final File outputDirectory;
    private final UiTester uiTester;

    @Inject
    public UiRunListener(UiTester uiTester, @OutputDirectory File outputDirectory)
    {
        this.uiTester = uiTester;
        this.outputDirectory = outputDirectory;
    }
    
    @Override
    public void testFailure(Failure failure) throws Exception
    {
        super.testFailure(failure);
        outputHtmlToFile(failure.getDescription().getDisplayName() + ".html", failure.getMessage());
    }

    private void outputHtmlToFile(String filename, String failureMessage) throws IOException
    {
        //create new directory if it's not already there
        outputDirectory.mkdirs();
        
        //output the results
        final File file = new File(outputDirectory, filename);
        Writer out = new OutputStreamWriter(new FileOutputStream(file));
        try
        {
            out.write(failureMessage != null ? failureMessage : "Failure");
            out.write(uiTester.getHtmlContents());
            System.err.println("Html captured to file: " + file.getAbsolutePath());
        }
        finally
        {
            closeQuietly(out);
        }
    }
}
