package com.atlassian.integrationtesting.ui;

import java.io.File;
import java.io.IOException;

import com.atlassian.integrationtesting.ui.CompositeUiTester.Backup;
import com.atlassian.integrationtesting.ui.CompositeUiTester.Login;
import com.atlassian.integrationtesting.ui.CompositeUiTester.WebSudoLogin;
import com.atlassian.integrationtesting.ui.UiTesters.BackupFile;

import com.gargoylesoftware.htmlunit.html.HtmlPage;
import com.google.common.base.Function;

import be.roam.hue.doj.Doj;

import static com.atlassian.integrationtesting.Functions.partiallyApply;
import static com.atlassian.integrationtesting.ui.UiTesters.doNothingWebSudoLogin;
import static com.atlassian.integrationtesting.ui.UiTesters.getHome;
import static com.atlassian.integrationtesting.ui.UiTesters.goToPageToLogout;
import static com.atlassian.integrationtesting.ui.UiTesters.isOnLogInPageByFormId;
import static com.atlassian.integrationtesting.ui.UiTesters.isOnLogInPageByFormName;
import static com.atlassian.integrationtesting.ui.UiTesters.logInByGoingTo;
import static com.atlassian.integrationtesting.ui.UiTesters.withJavascriptDisabled;
import static org.apache.commons.io.FileUtils.readFileToString;
import static org.apache.commons.io.FileUtils.toFile;
import static org.apache.commons.io.FileUtils.writeStringToFile;
import static org.apache.commons.lang.RandomStringUtils.randomAlphanumeric;

/**
 * <p>{@code UiTesterProvider}s for JIRA versions 4.1, 4.2 and 4.3.</p>
 * 
 * <p>The implementation of JIRAs restore functionality relies on a hidden ability called a "quick import".  We do this
 * because the regular import operation restarts the entire plugin system, which is a very time consuming operation.
 * This does have the caveat that any plugins that cache data internally will not be cleaned up automatically by the
 * plugin being restarted, which may result in it having stale data after the restore.  To avoid this, plugins should
 * listen for the {@code com.atlassian.jira.event.ClearCacheEvent} and flush their internal caches manually.</p>
 * 
 * <p>Note: We only use quick import for JIRA 4.1 and 4.2.  There is a bug in 4.3 where some of JIRA's internal caches
 * aren't being cleared properly so we must do the regular import operation.</p>
 */
public enum Jira implements UiTesterFunctionProvider
{
    v4_1
    {
        private final Function<Login, HtmlPage> logIn = logInByGoingTo("login.jsp").submitButtonId("login").build();
        public Function<Login, HtmlPage> logIn()
        {
            return logIn;
        }
        
        private final Function<UiTester, Boolean> isOnLogInPage = isOnLogInPageByFormName("loginform");
        public Function<UiTester, Boolean> isOnLogInPage()
        {
            return isOnLogInPage;
        }
        
        public Function<Backup, Void> restore()
        {
            return restoreData(quickImport());
        }
    },
    v4_2
    {
        private final Function<UiTester, Boolean> isOnLogInPage = isOnLogInPageByFormId("login-form");
        public Function<UiTester, Boolean> isOnLogInPage()
        {
            return isOnLogInPage;
        }
        
        public Function<Backup, Void> restore()
        {
            return restoreData(quickImport());
        }
    },
    v4_3
    {
        private final Function<UiTester, Boolean> isOnLogInPage = isOnLogInPageByFormId("login-form");
        
        public Function<UiTester, Boolean> isOnLogInPage()
        {
            return isOnLogInPage;
        }

        @Override
        public Function<WebSudoLogin, HtmlPage> webSudoLogIn()
        {
            return JiraWebSudoLogin.INSTANCE;
        }
        
        public Function<Backup, Void> restore()
        {
            return restoreData(regularImport());
        }
    };

    private final Function<Login, HtmlPage> logIn = logInByGoingTo("login.jsp").submitButtonId("login-form-submit").build();
    public Function<Login, HtmlPage> logIn()
    {
        return logIn;
    }

    private final Function<UiTester, Void> logout = new Function<UiTester, Void>()
    {
        final Function<UiTester, Void> goToLogOutPage = goToPageToLogout("secure/Logout!default.jspa");
        public Void apply(UiTester uiTester)
        {
            return withJavascriptDisabled(uiTester, partiallyApply(goToLogOutPage, uiTester));
        }
    };
    public Function<UiTester, Void> logout()
    {
        return logout;
    }

    public Function<WebSudoLogin, HtmlPage> webSudoLogIn()
    {
        return doNothingWebSudoLogin();
    }

    public Function<UiTester, String> getLoggedInUser()
    {
        return UiTesters.getLoggedInUser();
    }
    
    private static Function<Backup, Void> restoreData(final Function<BackupFile, Void> restore)
    {
        return  new Function<Backup, Void>()
        {
            final Function<Backup, Void> res = UiTesters.restore(ProcessBackupData.INSTANCE, restore);
    
            public Void apply(Backup backup)
            {
                return withJavascriptDisabled(backup.client, partiallyApply(res, backup));
            }
        };
    }
    
    private enum ProcessBackupData implements Function<Backup, File>
    {
        INSTANCE;

        public File apply(Backup backup)
        {
            try
            {
                String data = readFileToString(toFile(backup.data));
                data = data.replaceAll("@jira-home@", getHome());
                data = data.replaceAll("@base-url@", backup.client.getBaseUrl());
                File dataFile = new File(getHome(), "import/backup" + randomAlphanumeric(5) + ".xml");
                writeStringToFile(dataFile, data);
                return dataFile;
            }
            catch (IOException e)
            {
                throw new RestoreFromBackupException("Unable to process backup data file " + backup.data, e);
            }
        }
    }
    
    private static Function<BackupFile, Void> quickImport()
    {
        return Restore.QUICK;
    }
    
    private static Function<BackupFile, Void> regularImport()
    {
        return Restore.REGULAR;
    }
    
    private enum Restore implements Function<BackupFile, Void>
    {
        QUICK(true),
        REGULAR(false);

        private final boolean quickImport;
        
        private Restore(boolean quickImport)
        {
            this.quickImport = quickImport;
        }
        
        public Void apply(BackupFile backup)
        {
            try
            {
                backup.client.gotoPage("secure/admin/XmlRestore!default.jspa");
                backup.client.currentPage().getByAttribute("name", "filename").value(backup.file.getAbsolutePath());
                if (quickImport)
                {
                    backup.client.currentPage().getById("quickImport").check();
                }
                HtmlPage page = (HtmlPage) backup.client.currentPage().getById("restore_submit").click();
                Doj doj = Doj.on(page);
                Doj errorArea = doj.get("div.errorArea");
                if (!errorArea.isEmpty())
                {
                    String error = errorArea.get("ul li").text();
                    if (error.contains("You must edit jira-application.properties") && error.contains("jira.paths.set.allowed=true"))
                    {
                        throw new RestoreFromBackupException("Unable to restore because JIRA does not allow " +
                        		"restoring unless the jira.paths.set.allowed property in the " +
                        		"jira-application.properties file is set to true.  If you are using AMPS," +
                        		"you can customize the jira-application.properties file by creating an" +
                        		"\"overlay\" in your src/test/resources directory.  Create the directory " +
                        		"src/test/resources/{instance-id}-app/WEB-INF/classes - where {instance-id} is" +
                        		"replaced with the applications instance ID, typically just the application" +
                        		"name when running a single application in the test group - and place your" +
                        		"customized jira-applications.properties file in that directory.");
                    }
                    else
                    {
                        throw new RestoreFromBackupException(error);
                    }
                }
            }
            catch (IOException e)
            {
                throw new RestoreFromBackupException(e);
            }
            catch (ClassCastException e)
            {
                throw new RestoreFromBackupException("Element with id 'restore_submit' is not a button");
            }
            return null;
        }
    }

    private enum JiraWebSudoLogin implements Function<CompositeUiTester.WebSudoLogin, HtmlPage>
    {
        INSTANCE;

        public HtmlPage apply(CompositeUiTester.WebSudoLogin login)
        {
            // Lets kick in the webSudo
            login.client.gotoPage("secure/admin/XmlRestore!default.jspa");

             final Doj form = login.client.currentPage().get("form").first();
             // We are on the web sudo login page
             if (form != null && form.attribute("action").endsWith("WebSudoAuthenticate.jspa"))
             {

                 form.getById("login-form-authenticatePassword").value(login.password);
                 try
                 {
                     return (HtmlPage) form.get("input").withId("authenticateButton").click();
                 }
                 catch (IOException e)
                 {
                     throw new RuntimeException(e);
                 }
             }

             return (HtmlPage) login.client.currentPage().firstElement().getPage();
        }
    }
}
